package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.TGasUserInfo;
import com.zehong.system.mapper.TGasUserInfoMapper;
import com.zehong.system.service.ITGasUserInfoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 燃气用户Service业务层处理
 *
 * @author zehong
 * @date 2023-08-17
 */
@Service
public class TGasUserInfoServiceImpl implements ITGasUserInfoService
{
    private static final Logger log = LoggerFactory.getLogger(TGasUserInfoServiceImpl.class);

    @Autowired
    private TGasUserInfoMapper tGasUserInfoMapper;

    /**
     * 查询燃气用户
     *
     * @param gasUserId 燃气用户ID
     * @return 燃气用户
     */
    @Override
    public TGasUserInfo selectTGasUserInfoById(Long gasUserId)
    {
        return tGasUserInfoMapper.selectTGasUserInfoById(gasUserId);
    }

    /**
     * 查询燃气用户列表
     *
     * @param tGasUserInfo 燃气用户
     * @return 燃气用户
     */
    @Override
    public List<TGasUserInfo> selectTGasUserInfoList(TGasUserInfo tGasUserInfo)
    {
        return tGasUserInfoMapper.selectTGasUserInfoList(tGasUserInfo);
    }

    /**
     * 新增燃气用户
     *
     * @param tGasUserInfo 燃气用户
     * @return 结果
     */
    @Override
    public int insertTGasUserInfo(TGasUserInfo tGasUserInfo)
    {
        tGasUserInfo.setCreateTime(DateUtils.getNowDate());
        return tGasUserInfoMapper.insertTGasUserInfo(tGasUserInfo);
    }

    /**
     * 修改燃气用户
     *
     * @param tGasUserInfo 燃气用户
     * @return 结果
     */
    @Override
    public int updateTGasUserInfo(TGasUserInfo tGasUserInfo)
    {
        tGasUserInfo.setUpdateTime(DateUtils.getNowDate());
        return tGasUserInfoMapper.updateTGasUserInfo(tGasUserInfo);
    }

    /**
     * 批量删除燃气用户
     *
     * @param gasUserIds 需要删除的燃气用户ID
     * @return 结果
     */
    @Override
    public int deleteTGasUserInfoByIds(Long[] gasUserIds)
    {
        return tGasUserInfoMapper.deleteTGasUserInfoByIds(gasUserIds);
    }

    /**
     * 删除燃气用户信息
     *
     * @param gasUserId 燃气用户ID
     * @return 结果
     */
    @Override
    public int deleteTGasUserInfoById(Long gasUserId)
    {
        return tGasUserInfoMapper.deleteTGasUserInfoById(gasUserId);
    }

    /**
     * 燃气用户导入
     * @param tGasUserInfoList 燃气用户实体
     * @param updateSupport 是否更新
     * @return
     */
    @Override
    public String importTGasUserInfo(List<TGasUserInfo> tGasUserInfoList, boolean isUpdateSupport){
        if (StringUtils.isNull(tGasUserInfoList) || tGasUserInfoList.size() == 0){
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TGasUserInfo tGasUserInfo : tGasUserInfoList){
            try {
                TGasUserInfo queryGasUserInfo = new TGasUserInfo();
                queryGasUserInfo.setGasUserName(tGasUserInfo.getGasUserName());
                List<TGasUserInfo> queryGasUserInfoList = tGasUserInfoMapper.selectTGasUserInfoList(queryGasUserInfo);
                if (StringUtils.isNull(queryGasUserInfoList) || queryGasUserInfoList.size() == 0){
                    this.insertTGasUserInfo(tGasUserInfo);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、燃气用户名称 " + tGasUserInfo.getGasUserName() + " 导入成功");
                }else if (isUpdateSupport){
                    tGasUserInfo.setGasUserId(queryGasUserInfoList.get(0).getGasUserId());
                    this.updateTGasUserInfo(tGasUserInfo);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、燃气用户名称 " + tGasUserInfo.getGasUserName() + " 更新成功");
                }else{
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、燃气用户名称 " + tGasUserInfo.getGasUserName() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、燃气用户名称 " + tGasUserInfo.getGasUserName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 查询燃气用户信息
     * @param tGasUserInfo
     * @return
     */
    @Override
    public List<TGasUserInfo> gasUser(TGasUserInfo tGasUserInfo) {
        return tGasUserInfoMapper.gasUser(tGasUserInfo);
    }
}
