package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.BottleStatistics;
import com.zehong.system.domain.TGasBottleInfo;
import com.zehong.system.domain.TGasStorageStationInfo;
import com.zehong.system.mapper.TGasBottleInfoMapper;
import com.zehong.system.mapper.TGasStorageStationInfoMapper;
import com.zehong.system.service.ITGasBottleInfoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 气瓶信息Service业务层处理
 *
 * @author zehong
 * @date 2023-08-15
 */
@Service
public class TGasBottleInfoServiceImpl implements ITGasBottleInfoService
{
    private static final Logger log = LoggerFactory.getLogger(TGasBottleInfoServiceImpl.class);

    @Autowired
    private TGasBottleInfoMapper tGasBottleInfoMapper;

    @Resource
    private TGasStorageStationInfoMapper tGasStorageStationInfoMapper;

    /**
     * 查询气瓶信息
     *
     * @param bottleId 气瓶信息ID
     * @return 气瓶信息
     */
    @Override
    public TGasBottleInfo selectTGasBottleInfoById(Long bottleId)
    {
        return tGasBottleInfoMapper.selectTGasBottleInfoById(bottleId);
    }

    /**
     * 查询气瓶信息列表
     *
     * @param tGasBottleInfo 气瓶信息
     * @return 气瓶信息
     */
    @Override
    public List<TGasBottleInfo> selectTGasBottleInfoList(TGasBottleInfo tGasBottleInfo)
    {
        return tGasBottleInfoMapper.selectTGasBottleInfoList(tGasBottleInfo);
    }

    /**
     * 新增气瓶信息
     *
     * @param tGasBottleInfo 气瓶信息
     * @return 结果
     */
    @Override
    public int insertTGasBottleInfo(TGasBottleInfo tGasBottleInfo)
    {
        tGasBottleInfo.setCreateTime(DateUtils.getNowDate());
        return tGasBottleInfoMapper.insertTGasBottleInfo(tGasBottleInfo);
    }

    /**
     * 修改气瓶信息
     *
     * @param tGasBottleInfo 气瓶信息
     * @return 结果
     */
    @Override
    public int updateTGasBottleInfo(TGasBottleInfo tGasBottleInfo)
    {
        tGasBottleInfo.setUpdateTime(DateUtils.getNowDate());
        return tGasBottleInfoMapper.updateTGasBottleInfo(tGasBottleInfo);
    }

    /**
     * 批量删除气瓶信息
     *
     * @param bottleIds 需要删除的气瓶信息ID
     * @return 结果
     */
    @Override
    public int deleteTGasBottleInfoByIds(Long[] bottleIds)
    {
        return tGasBottleInfoMapper.deleteTGasBottleInfoByIds(bottleIds);
    }

    /**
     * 删除气瓶信息信息
     *
     * @param bottleId 气瓶信息ID
     * @return 结果
     */
    @Override
    public int deleteTGasBottleInfoById(Long bottleId)
    {
        return tGasBottleInfoMapper.deleteTGasBottleInfoById(bottleId);
    }

    /**
     * 气瓶数据导出
     * @param tGasBottleInfoList 气瓶实体
     * @param isUpdateSupport 是否更新
     * @return
     */
    @Override
    public String importTGasBottleInfo(List<TGasBottleInfo> tGasBottleInfoList, Boolean isUpdateSupport){
        if (StringUtils.isNull(tGasBottleInfoList) || tGasBottleInfoList.size() == 0){
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TGasBottleInfo tGasBottleInfo : tGasBottleInfoList){
            try {
                //查询储配站信息
                TGasStorageStationInfo queryGasStorageStationInfo = new TGasStorageStationInfo();
                queryGasStorageStationInfo.setStationName(tGasBottleInfo.getStationName());
                List<TGasStorageStationInfo> gasStorageStationInfo = tGasStorageStationInfoMapper.selectTGasStorageStationInfoList(queryGasStorageStationInfo);
                if(CollectionUtils.isEmpty(gasStorageStationInfo)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、气瓶条码为 " + tGasBottleInfo.getBottleCode() +"、储配站" + tGasBottleInfo.getStationName() + " 不存在请创建或导入");
                    continue;
                }
                TGasBottleInfo gasBottleInfo = new TGasBottleInfo();
                gasBottleInfo.setBottleCode(tGasBottleInfo.getBottleCode());
                List<TGasBottleInfo> queryGasBottleInfo = tGasBottleInfoMapper.selectTGasBottleInfoList(gasBottleInfo);
                if (StringUtils.isNull(queryGasBottleInfo) || queryGasBottleInfo.size() == 0){
                    tGasBottleInfo.setStationId(gasStorageStationInfo.get(0).getStationId());
                    this.insertTGasBottleInfo(tGasBottleInfo);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、气瓶条码 " + tGasBottleInfo.getBottleCode() + " 导入成功");
                }else if (isUpdateSupport){
                    queryGasBottleInfo.stream().forEach(bottleInfo ->{
                        tGasBottleInfo.setBottleId(bottleInfo.getBottleId());
                        tGasBottleInfo.setStationId(gasStorageStationInfo.get(0).getStationId());
                        this.updateTGasBottleInfo(tGasBottleInfo);
                    });
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、气瓶条码 " + tGasBottleInfo.getBottleCode() + " 更新成功");
                }else{
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、气瓶条码 " + tGasBottleInfo.getBottleCode() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、气瓶条码 " + tGasBottleInfo.getBottleCode() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 气瓶统计
     * @param stationId 储配站id
     * @return
     */
    @Override
    public List<BottleStatistics> bottleStatistics(Long stationId){
        return tGasBottleInfoMapper.bottleStatistics(stationId);
    }

    /**
     * 查询气瓶详细信息
     * @param bottleId
     * @return
     */
    @Override
    public TGasBottleInfo getInf(String bottleId) {
        return tGasBottleInfoMapper.getInf(bottleId);
    }

    @Override
    public TGasBottleInfo getInfn(String bottleId) {
        return tGasBottleInfoMapper.getInfn(bottleId);
    }
}
