package com.zehong.system.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.exception.BaseException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.SecurityUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.common.utils.bean.BeanUtils;
import com.zehong.common.utils.uuid.UUID;
import com.zehong.system.domain.vo.TEnterpriseInfoVO;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.zehong.system.mapper.TEnterpriseInfoMapper;
import com.zehong.system.domain.TEnterpriseInfo;
import com.zehong.system.service.ITEnterpriseInfoService;

/**
 * 企业信息Service业务层处理
 * 
 * @author zehong
 * @date 2022-01-24
 */
@Service
public class TEnterpriseInfoServiceImpl implements ITEnterpriseInfoService 
{
    @Autowired
    private TEnterpriseInfoMapper tEnterpriseInfoMapper;

    /**
     * 查询企业信息
     * 
     * @param enterpriseId 企业信息ID
     * @return 企业信息
     */
    @Override
    public TEnterpriseInfo selectTEnterpriseInfoById(String enterpriseId)
    {
        return tEnterpriseInfoMapper.selectTEnterpriseInfoById(enterpriseId);
    }

    /**
     * 查询企业信息列表
     *
     * @param tEnterpriseInfo 企业信息
     * @return 企业信息
     */
    @Override
    public List<TEnterpriseInfoVO> enterpriseInfoList(TEnterpriseInfo tEnterpriseInfo)
    {
        List<TEnterpriseInfoVO> list = new ArrayList<>();
        List<TEnterpriseInfo> enterpriseInfoList = tEnterpriseInfoMapper.selectTEnterpriseInfoList(tEnterpriseInfo);

        for(TEnterpriseInfo info : enterpriseInfoList){
            TEnterpriseInfoVO tEnterpriseInfoVO = new TEnterpriseInfoVO();
            BeanUtils.copyProperties(info, tEnterpriseInfoVO);
            tEnterpriseInfoVO.setCompanyType(info.getEnterpriseId().toString());
            list.add(tEnterpriseInfoVO);
        }
        return list;
    }

    /**
     * 查询企业信息列表
     * 
     * @param tEnterpriseInfo 企业信息
     * @return 企业信息
     */
    @Override
    public List<TEnterpriseInfo> selectTEnterpriseInfoList(TEnterpriseInfo tEnterpriseInfo)
    {
        return tEnterpriseInfoMapper.selectTEnterpriseInfoList(tEnterpriseInfo);
    }


    /**
     * 无条件查询所有企业
     * @return re
     */
    @Override
    public List<TEnterpriseInfo> queryAllEnterprise() {
        return tEnterpriseInfoMapper.queryAllEnterprise();
    }

    /**
     * 新增企业信息
     * 
     * @param tEnterpriseInfo 企业信息
     * @return 结果
     */
    @Override
    public int insertTEnterpriseInfo(TEnterpriseInfo tEnterpriseInfo)
    {
        tEnterpriseInfo.setCreateTime(DateUtils.getNowDate());
        return tEnterpriseInfoMapper.insertTEnterpriseInfo(tEnterpriseInfo);
    }

    /**
     * 修改企业信息
     * 
     * @param tEnterpriseInfo 企业信息
     * @return 结果
     */
    @Override
    public int updateTEnterpriseInfo(TEnterpriseInfo tEnterpriseInfo)
    {
        tEnterpriseInfo.setUpdateTime(DateUtils.getNowDate());
        return tEnterpriseInfoMapper.updateTEnterpriseInfo(tEnterpriseInfo);
    }

    /**
     * 批量删除企业信息
     * 
     * @param enterpriseIds 需要删除的企业信息ID
     * @return 结果
     */
    @Override
    public int deleteTEnterpriseInfoByIds(String[] enterpriseIds)
    {
        return tEnterpriseInfoMapper.deleteTEnterpriseInfoByIds(enterpriseIds);
    }

    /**
     * 删除企业信息信息
     * 
     * @param enterpriseId 企业信息ID
     * @return 结果
     */
    @Override
    public int deleteTEnterpriseInfoById(String enterpriseId)
    {
        return tEnterpriseInfoMapper.deleteTEnterpriseInfoById(enterpriseId);
    }

    /**
     * 导入企业信息
     * @param tEnterpriseInfos 用户数据列表
     * @param isUpdateSupport 是否更新支持，如果已存在，则进行更新数据
     * @param operName 操作用户
     * @return
     */
    @Override
    public String importEnterpriseInfo(List<TEnterpriseInfo> tEnterpriseInfos, Boolean isUpdateSupport, String operName) {
        if (CollectionUtils.isEmpty(tEnterpriseInfos)) {
            throw new BaseException("导入项目编号数据不能为空！ ");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TEnterpriseInfo enterpriseInfo : tEnterpriseInfos) {
            try {
                checkUploadRequired(enterpriseInfo);
                SysUser user = SecurityUtils.getLoginUser().getUser();
                enterpriseInfo.setCreateEnterpriseId(user.getDeptId());
                enterpriseInfo.setEnterpriseId(UUID.randomUUID().toString());
                this.insertTEnterpriseInfo(enterpriseInfo);
                successNum++;
                successMsg.append("<br/>" + successNum + "、项目编号 " + enterpriseInfo.getEnterpriseName() + " 导入成功");
            } catch (Exception e) {
                failureNum++;
                String msg = "<br/>" + failureNum + "、项目编号 " + enterpriseInfo.getEnterpriseName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
            }
        }
        if (failureNum > 0) {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new BaseException(failureMsg.toString());
        } else {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 校验导入必填
     * @param tEnterpriseInfo t
     */
    public void checkUploadRequired(TEnterpriseInfo tEnterpriseInfo) {
        if (tEnterpriseInfo.getEnterpriseName() == null || "".equals(tEnterpriseInfo.getEnterpriseName())) {
            throw new BaseException("企业名称必填！！！");
        }

        if (tEnterpriseInfo.getEnterpriseType() == null) {
            throw new BaseException("企业类型必传！！！");
        }

        if (tEnterpriseInfo.getRegisterAddress() == null || "".equals(tEnterpriseInfo.getRegisterAddress())) {
            throw new BaseException("注册地址必传！！！");
        }

        if (tEnterpriseInfo.getLegalRepresentative() == null || "".equals(tEnterpriseInfo.getLegalRepresentative())) {
            throw new BaseException("法定代表人必传！！！");
        }
    }
}
