package com.zehong.system.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.GovernmentDataCopyUtil;
import com.zehong.common.utils.GovernmentDataUtil;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.TInsHazRef;
import com.zehong.system.domain.vo.TInsHazRefVo;
import com.zehong.system.mapper.TInsHazRefMapper;
import com.zehong.system.service.ITInsHazRefService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.sql.SQLIntegrityConstraintViolationException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 监督检查-监督检查发现隐患及整改，执法结果Service业务层处理
 * 
 * @author zehong
 * @date 2024-05-21
 */
@Service
public class TInsHazRefServiceImpl implements ITInsHazRefService 
{
    private static final Logger log = LoggerFactory.getLogger(TInsHazRefServiceImpl.class);

    @Autowired
    private TInsHazRefMapper tInsHazRefMapper;

    @Resource
    private GovernmentDataUtil governmentDataUtil;

    /**
     * 查询监督检查-监督检查发现隐患及整改，执法结果
     * 
     * @param fInsHazRefId 监督检查-监督检查发现隐患及整改，执法结果ID
     * @return 监督检查-监督检查发现隐患及整改，执法结果
     */
    @Override
    public TInsHazRef selectTInsHazRefById(Long fInsHazRefId)
    {
        return tInsHazRefMapper.selectTInsHazRefById(fInsHazRefId);
    }

    /**
     * 查询监督检查-监督检查发现隐患及整改，执法结果列表
     * 
     * @param tInsHazRef 监督检查-监督检查发现隐患及整改，执法结果
     * @return 监督检查-监督检查发现隐患及整改，执法结果
     */
    @Override
    public List<TInsHazRef> selectTInsHazRefList(TInsHazRef tInsHazRef)
    {
        return tInsHazRefMapper.selectTInsHazRefList(tInsHazRef);
    }

    /**
     * 新增监督检查-监督检查发现隐患及整改，执法结果
     * 
     * @param tInsHazRef 监督检查-监督检查发现隐患及整改，执法结果
     * @return 结果
     */
    @Override
    public int insertTInsHazRef(TInsHazRef tInsHazRef)
    {
        try{
            tInsHazRef.setfLastTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
            return tInsHazRefMapper.insertTInsHazRef(tInsHazRef);
        }catch (Exception  e){
            Throwable cause = e.getCause();
            if (cause instanceof SQLIntegrityConstraintViolationException){
                String errMsg = cause.getMessage();
                if (StringUtils.isNotEmpty(errMsg) && errMsg.contains("index_hazard_unique_code")){
                    throw new CustomException("隐患唯一编码不唯一");
                }
            }
            return 0;
        }
    }

    /**
     * 修改监督检查-监督检查发现隐患及整改，执法结果
     * 
     * @param tInsHazRef 监督检查-监督检查发现隐患及整改，执法结果
     * @return 结果
     */
    @Override
    public int updateTInsHazRef(TInsHazRef tInsHazRef)
    {
        try{
            tInsHazRef.setfLastTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
            return tInsHazRefMapper.updateTInsHazRef(tInsHazRef);
        }catch (Exception  e){
            Throwable cause = e.getCause();
            if (cause instanceof SQLIntegrityConstraintViolationException){
                String errMsg = cause.getMessage();
                if (StringUtils.isNotEmpty(errMsg) && errMsg.contains("index_hazard_unique_code")){
                    throw new CustomException("隐患唯一编码不唯一");
                }
            }
            return 0;
        }

    }

    /**
     * 批量删除监督检查-监督检查发现隐患及整改，执法结果
     * 
     * @param fInsHazRefIds 需要删除的监督检查-监督检查发现隐患及整改，执法结果ID
     * @return 结果
     */
    @Override
    public int deleteTInsHazRefByIds(Long[] fInsHazRefIds)
    {
        return tInsHazRefMapper.deleteTInsHazRefByIds(fInsHazRefIds);
    }

    /**
     * 删除监督检查-监督检查发现隐患及整改，执法结果信息
     * 
     * @param fInsHazRefId 监督检查-监督检查发现隐患及整改，执法结果ID
     * @return 结果
     */
    @Override
    public int deleteTInsHazRefById(Long fInsHazRefId)
    {
        return tInsHazRefMapper.deleteTInsHazRefById(fInsHazRefId);
    }

    @Override
    public int reportHazRefInfo(Long fInsHazRefId) throws Exception {
        TInsHazRef insHazRef = tInsHazRefMapper.selectTInsHazRefById(fInsHazRefId);
        TInsHazRefVo insHazRefVo = new TInsHazRefVo();
        GovernmentDataCopyUtil.copyToGovernData(insHazRef,insHazRefVo);
        List<TInsHazRefVo> data = new ArrayList<>();
        data.add(insHazRefVo);
        JSONObject reportResult = governmentDataUtil.setInfo("inspection/record/information","WRITE",data);
        log.info("上传监督检查发现隐患及整改、执法结果===================" + reportResult.toJSONString());
        if(!"0".equals(reportResult.getString("resultCode"))) throw new CustomException("上传监督检查发现隐患及整改、执法结果市局接口失败");
        return 1;
    }


    /**
     * 根据检查记录上传隐患
     * @param checkCode
     * @return
     */
    @Override
    public int reportHazRefInfoByCheckCode(String checkCode) throws Exception {
        TInsHazRef tInsHazRef = new TInsHazRef();
        tInsHazRef.setfCheckCode(checkCode);
        List<TInsHazRef> refList = tInsHazRefMapper.selectTInsHazRefList(tInsHazRef);
        if(CollectionUtils.isEmpty(refList)) throw new CustomException("该检查记录下未查到关联隐患");
        List<TInsHazRefVo> data = new ArrayList<>();
        for(TInsHazRef insHazRef : refList){
            TInsHazRefVo insHazRefVo = new TInsHazRefVo();
            GovernmentDataCopyUtil.copyToGovernData(insHazRef,insHazRefVo);
            data.add(insHazRefVo);
        }
        JSONObject reportResult = governmentDataUtil.setInfo("inspection/record/information","WRITE",data);
        log.info("上传监督检查发现隐患及整改、执法结果===================" + reportResult.toJSONString());
        if(!"0".equals(reportResult.getString("resultCode"))) throw new CustomException("上传隐患市局接口失败");
        return 1;
    }

}
