package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.BaseInfoStatistics;
import com.zehong.system.domain.TGasStorageStationInfo;
import com.zehong.system.mapper.TGasStorageStationInfoMapper;
import com.zehong.system.service.ITGasStorageStationInfoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 储配站信息Service业务层处理
 * 
 * @author zehong
 * @date 2023-08-14
 */
@Service
public class TGasStorageStationInfoServiceImpl implements ITGasStorageStationInfoService 
{
    private static final Logger log = LoggerFactory.getLogger(TGasStorageStationInfoServiceImpl.class);

    @Autowired
    private TGasStorageStationInfoMapper tGasStorageStationInfoMapper;

    /**
     * 查询储配站信息
     * 
     * @param stationId 储配站信息ID
     * @return 储配站信息
     */
    @Override
    public TGasStorageStationInfo selectTGasStorageStationInfoById(Long stationId)
    {
        return tGasStorageStationInfoMapper.selectTGasStorageStationInfoById(stationId);
    }

    /**
     * 查询储配站信息列表
     * 
     * @param tGasStorageStationInfo 储配站信息
     * @return 储配站信息
     */
    @Override
    public List<TGasStorageStationInfo> selectTGasStorageStationInfoList(TGasStorageStationInfo tGasStorageStationInfo)
    {
        return tGasStorageStationInfoMapper.selectTGasStorageStationInfoList(tGasStorageStationInfo);
    }

    /**
     * 新增储配站信息
     * 
     * @param tGasStorageStationInfo 储配站信息
     * @return 结果
     */
    @Override
    public int insertTGasStorageStationInfo(TGasStorageStationInfo tGasStorageStationInfo)
    {
        tGasStorageStationInfo.setCreateTime(DateUtils.getNowDate());
        return tGasStorageStationInfoMapper.insertTGasStorageStationInfo(tGasStorageStationInfo);
    }

    /**
     * 修改储配站信息
     * 
     * @param tGasStorageStationInfo 储配站信息
     * @return 结果
     */
    @Override
    public int updateTGasStorageStationInfo(TGasStorageStationInfo tGasStorageStationInfo)
    {
        tGasStorageStationInfo.setUpdateTime(DateUtils.getNowDate());
        return tGasStorageStationInfoMapper.updateTGasStorageStationInfo(tGasStorageStationInfo);
    }

    /**
     * 批量删除储配站信息
     * 
     * @param stationIds 需要删除的储配站信息ID
     * @return 结果
     */
    @Override
    public int deleteTGasStorageStationInfoByIds(Long[] stationIds)
    {
        return tGasStorageStationInfoMapper.deleteTGasStorageStationInfoByIds(stationIds);
    }

    /**
     * 删除储配站信息信息
     * 
     * @param stationId 储配站信息ID
     * @return 结果
     */
    @Override
    public int deleteTGasStorageStationInfoById(Long stationId)
    {
        return tGasStorageStationInfoMapper.deleteTGasStorageStationInfoById(stationId);
    }

    /**
     * 导入数据
     *
     * @param tGasStorageStationInfoList 数据列表
     * @param isUpdateSupport 是否更新支持，如果已存在，则进行更新数据
     * @return 结果
     */
    @Override
    public String importTGasStorageStationInfo(List<TGasStorageStationInfo> tGasStorageStationInfoList, Boolean isUpdateSupport){
        if (StringUtils.isNull(tGasStorageStationInfoList) || tGasStorageStationInfoList.size() == 0){
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TGasStorageStationInfo tGasStorageStationInfo : tGasStorageStationInfoList)
        {
            try {
                TGasStorageStationInfo queryGasStorageStationInfo = new TGasStorageStationInfo();
                queryGasStorageStationInfo.setStationName(tGasStorageStationInfo.getStationName());
                List<TGasStorageStationInfo> gasStorageStationInfos = tGasStorageStationInfoMapper.selectTGasStorageStationInfoList(queryGasStorageStationInfo);
                if (StringUtils.isNull(gasStorageStationInfos) || gasStorageStationInfos.size() == 0){
                    this.insertTGasStorageStationInfo(tGasStorageStationInfo);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、储配站 " + tGasStorageStationInfo.getStationName() + " 导入成功");
                }
                else if (isUpdateSupport){
                    gasStorageStationInfos.stream().forEach(stationInfo ->{
                        tGasStorageStationInfo.setStationId(stationInfo.getStationId());
                        this.updateTGasStorageStationInfo(tGasStorageStationInfo);
                    });
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、储配站 " + tGasStorageStationInfo.getStationName() + " 更新成功");
                }else{
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、储配站 " + tGasStorageStationInfo.getStationName() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、储配站 " + tGasStorageStationInfo.getStationName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

     /**
     * 基础信息统计
     * @param stationId 储配站id
     * @return
     */
     @Override
     public List<BaseInfoStatistics> baseInfoStatistics(Long stationId){
         return tGasStorageStationInfoMapper.baseInfoStatistics(stationId);
     }
}
