package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.*;
import com.zehong.system.domain.vo.AirChargeStatisticsVo;
import com.zehong.system.mapper.TAirChargeRecordMapper;
import com.zehong.system.mapper.TGasBottleInfoMapper;
import com.zehong.system.mapper.TGasStorageStationInfoMapper;
import com.zehong.system.mapper.TPractitionerInfoMapper;
import com.zehong.system.service.ITAirChargeRecordService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 充装记录Service业务层处理
 * 
 * @author zehong
 * @date 2023-08-21
 */
@Service
public class TAirChargeRecordServiceImpl implements ITAirChargeRecordService 
{
    private static final Logger log = LoggerFactory.getLogger(TAirChargeRecordServiceImpl.class);

    @Autowired
    private TAirChargeRecordMapper tAirChargeRecordMapper;

    @Resource
    private TGasStorageStationInfoMapper tGasStorageStationInfoMapper;

    @Resource
    private TGasBottleInfoMapper tGasBottleInfoMapper;

    @Resource
    private TPractitionerInfoMapper tPractitionerInfoMapper;

    /**
     * 查询充装记录
     * 
     * @param chargeRecordId 充装记录ID
     * @return 充装记录
     */
    @Override
    public TAirChargeRecord selectTAirChargeRecordById(Long chargeRecordId)
    {
        return tAirChargeRecordMapper.selectTAirChargeRecordById(chargeRecordId);
    }

    /**
     * 查询充装记录列表
     * 
     * @param tAirChargeRecord 充装记录
     * @return 充装记录
     */
    @Override
    public List<TAirChargeRecord> selectTAirChargeRecordList(TAirChargeRecord tAirChargeRecord)
    {
        return tAirChargeRecordMapper.selectTAirChargeRecordList(tAirChargeRecord);
    }

    /**
     * 新增充装记录
     * 
     * @param tAirChargeRecord 充装记录
     * @return 结果
     */
    @Override
    public int insertTAirChargeRecord(TAirChargeRecord tAirChargeRecord)
    {
        tAirChargeRecord.setCreateTime(DateUtils.getNowDate());
        return tAirChargeRecordMapper.insertTAirChargeRecord(tAirChargeRecord);
    }

    /**
     * 修改充装记录
     * 
     * @param tAirChargeRecord 充装记录
     * @return 结果
     */
    @Override
    public int updateTAirChargeRecord(TAirChargeRecord tAirChargeRecord)
    {
        tAirChargeRecord.setUpdateTime(DateUtils.getNowDate());
        return tAirChargeRecordMapper.updateTAirChargeRecord(tAirChargeRecord);
    }

    /**
     * 批量删除充装记录
     * 
     * @param chargeRecordIds 需要删除的充装记录ID
     * @return 结果
     */
    @Override
    public int deleteTAirChargeRecordByIds(Long[] chargeRecordIds)
    {
        return tAirChargeRecordMapper.deleteTAirChargeRecordByIds(chargeRecordIds);
    }

    /**
     * 删除充装记录信息
     * 
     * @param chargeRecordId 充装记录ID
     * @return 结果
     */
    @Override
    public int deleteTAirChargeRecordById(Long chargeRecordId)
    {
        return tAirChargeRecordMapper.deleteTAirChargeRecordById(chargeRecordId);
    }

    /**
     * 充装记录导入
     * @param airChargeRecordList 充装记录实体
     * @param isUpdateSupport 是否更新
     * @return
     */
    @Override
    public String importAirChargeRecordInfo(List<TAirChargeRecord> airChargeRecordList, boolean isUpdateSupport){
        if (StringUtils.isNull(airChargeRecordList) || airChargeRecordList.size() == 0){
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TAirChargeRecord tAirChargeRecord : airChargeRecordList){
            try {
                //查询储配站信息
                TGasStorageStationInfo queryGasStorageStationInfo = new TGasStorageStationInfo();
                queryGasStorageStationInfo.setStationName(tAirChargeRecord.getStationName());
                //储配站
                List<TGasStorageStationInfo> gasStorageStationInfo = tGasStorageStationInfoMapper.selectTGasStorageStationInfoList(queryGasStorageStationInfo);
                if(CollectionUtils.isEmpty(gasStorageStationInfo)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、气瓶条码 " + tAirChargeRecord.getBottleCode() +"、储配站" + tAirChargeRecord.getStationName() + " 不存在请创建或导入");
                    continue;
                }

                //气瓶
                TGasBottleInfo tGasBottleInfo = new TGasBottleInfo();
                tGasBottleInfo.setBottleCode(tAirChargeRecord.getBottleCode());
                List<TGasBottleInfo> gasBottleInfoList = tGasBottleInfoMapper.selectTGasBottleInfoList(tGasBottleInfo);
                if(CollectionUtils.isEmpty(gasBottleInfoList)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、气瓶条码 " + tAirChargeRecord.getBottleCode() + " 不存在请创建");
                    continue;
                }
                //从业人员
                TPractitionerInfo query = new TPractitionerInfo();
                query.setName(tAirChargeRecord.getChargeOperatorName());
                List<TPractitionerInfo> queryPractitionerInfo = tPractitionerInfoMapper.selectTPractitionerInfoList(query);
                if(CollectionUtils.isEmpty(queryPractitionerInfo)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、从业人员 " + tAirChargeRecord.getChargeOperatorName() + " 不存在请创建");
                    continue;
                }

                tAirChargeRecord.setStationId(gasStorageStationInfo.get(0).getStationId());
                tAirChargeRecord.setBottleId(gasBottleInfoList.get(0).getBottleId());
                tAirChargeRecord.setChargeOperator(queryPractitionerInfo.get(0).getPractitionerId());
                this.insertTAirChargeRecord(tAirChargeRecord);
                successNum++;
                successMsg.append("<br/>" + successNum + "、气瓶条码为 " + tAirChargeRecord.getBottleCode() + " 充装记录导入成功");
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、气瓶条码为 " + tAirChargeRecord.getBottleCode() + " 充装记录导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 储配站充装统计
     * @param airChargeStatisticsVo 统计实体
     * @return
     */
    @Override
    public List<AirChargeStationStatistics> airChargeStationStatistics(AirChargeStatisticsVo airChargeStatisticsVo){
        return tAirChargeRecordMapper.airChargeStationStatistics(airChargeStatisticsVo);
    }

    /**
     * 充装人员统计
     * @param airChargeStatisticsVo 统计
     * @return
     */
    @Override
    public List<AirChargeOperatorStatistics> airChargeOperatorStatistics(AirChargeStatisticsVo airChargeStatisticsVo){
        return tAirChargeRecordMapper.airChargeOperatorStatistics(airChargeStatisticsVo);
    }
}
