package com.zehong.web.controller.supervise;

import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.core.page.TableDataInfo;
import com.zehong.common.enums.BusinessType;
import com.zehong.common.enums.ResultEnum;
import com.zehong.common.exception.IOCException;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.system.domain.TDetectorInfo;
import com.zehong.system.domain.vo.TDetectorAlarmInfoVO;
import com.zehong.system.service.ITDetectorInfoService;
import com.zehong.system.service.ITDetectorReportDataService;
import com.zehong.system.service.ITDetectorUserService;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 探测器Controller
 * 
 * @author zehong
 * @date 2021-11-02
 */
@RestController
@RequestMapping("/detector/detectorInfo")
@Slf4j
public class TDetectorInfoController extends BaseController
{
    @Autowired
    private ITDetectorInfoService tDetectorInfoService;
    @Autowired
    private ITDetectorUserService tDetectorUserService;
    @Autowired
    private ITDetectorReportDataService tDetectorReportDataService;

     /**
     * 查询探测器列表
     */
    @GetMapping("/list")
    public TableDataInfo list(TDetectorInfo tDetectorInfo)
    {
        startPage();
        List<TDetectorInfo> list = tDetectorInfoService.selectTDetectorInfoList(tDetectorInfo);
        return getDataTable(list);
    }

    /**
     * 获取探测器列表
     * @param tDetectorInfo
     * @return
     */
    @GetMapping("/detectorInfoList")
    public AjaxResult detectorInfoList(TDetectorInfo tDetectorInfo){
        return AjaxResult.success(tDetectorInfoService.selectTDetectorInfoList(tDetectorInfo));
    }

    /**
     * 导出探测器列表
     */
    @Log(title = "探测器", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TDetectorInfo tDetectorInfo)
    {
        List<TDetectorInfo> list = tDetectorInfoService.selectTDetectorInfoList(tDetectorInfo);
        ExcelUtil<TDetectorInfo> util = new ExcelUtil<TDetectorInfo>(TDetectorInfo.class);
        return util.exportExcel(list, "探测器数据");
    }

    /**
     * 获取探测器详细信息
     */
    @GetMapping(value = "/{detectorId}")
    public AjaxResult getInfo(@PathVariable("detectorId") Long detectorId)
    {
        return AjaxResult.success(tDetectorInfoService.selectTDetectorInfoById(detectorId));
    }

    /**
     * 新增探测器
     */
    @Log(title = "探测器", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TDetectorInfo tDetectorInfo)
    {
        return toAjax(tDetectorInfoService.insertTDetectorInfo(tDetectorInfo));
    }

    /**
     * 修改探测器
     */
    @Log(title = "探测器", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TDetectorInfo tDetectorInfo)
    {
        return toAjax(tDetectorInfoService.updateTDetectorInfo(tDetectorInfo));
    }

    /**
     * 删除探测器
     */
    @Log(title = "探测器", businessType = BusinessType.DELETE)
	@DeleteMapping("/{detectorIds}")
    public AjaxResult remove(@PathVariable Long[] detectorIds)
    {
        return toAjax(tDetectorInfoService.deleteTDetectorInfoByIds(detectorIds));
    }

    @ApiOperation(value = "大屏数据初始化时主动查询的感知设备报警信息前50条")
    @GetMapping("/getAlarmData")
    public AjaxResult getAlarmData(){
        ModelMap map = new ModelMap();
        try {
            TDetectorInfo detector = new TDetectorInfo();
            detector.setIsDel("0");
            detector.setDetectorStatus("0");
            List<TDetectorInfo> onlineList = tDetectorInfoService.selectTDetectorInfoList(detector);
            int onlineNum = onlineList.size();

            detector.setDetectorStatus("1");
            List<TDetectorInfo> offlineList = tDetectorInfoService.selectTDetectorInfoList(detector);
            int offlineNum = offlineList.size();

            detector.setDetectorStatus("2");
            List<TDetectorInfo> alarmList = tDetectorInfoService.selectTDetectorInfoList(detector);
            int alarmNum = alarmList.size();

            List<TDetectorAlarmInfoVO> alarmVO = tDetectorReportDataService.selectTDetectorAlarm();
            for(TDetectorAlarmInfoVO alarm : alarmVO){
                if("1".equals(alarm.getDetectorType())){
                    alarm.setDetectorType("家用探测器");
                } else if ("2".equals(alarm.getDetectorType())){
                    alarm.setDetectorType("工业探测器");
                }else if ("3".equals(alarm.getDetectorType())){
                    alarm.setDetectorType("可燃气体探测器");
                }else if ("4".equals(alarm.getDetectorType())){
                    alarm.setDetectorType("激光家报");
                }else if ("5".equals(alarm.getDetectorType())){
                    alarm.setDetectorType("地下井探测器");
                }
            }

            map.put("alarmNum", alarmNum);
            map.put("onlineNum", onlineNum);
            map.put("offlineNum", offlineNum);
            map.put("pageData", alarmVO);

        } catch (Exception ie){
            ie.printStackTrace();
            log.error("【查询报警数据】操作出错,error={}",ie.getMessage());
            throw new IOCException(ResultEnum.OPERATION_FAIL.getCode(), "查询报警数据操作出错");
        }

        return AjaxResult.success(map);
    }
    /**
     * 获取探测器监控数据
     * @param tDetectorInfo
     * @return
     */
    @GetMapping("/selectDetectorMointor")
    public TableDataInfo selectDetectorMointor(TDetectorInfo tDetectorInfo){
        startPage();
        List<TDetectorInfo> list = tDetectorInfoService.selectDetectorMointor(tDetectorInfo);
        return  getDataTable(list);
    }
}
