package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.SysPost;
import com.zehong.system.domain.TGasStorageStationInfo;
import com.zehong.system.domain.TPractitionerInfo;
import com.zehong.system.mapper.SysPostMapper;
import com.zehong.system.mapper.TGasStorageStationInfoMapper;
import com.zehong.system.mapper.TPractitionerInfoMapper;
import com.zehong.system.service.ITPractitionerInfoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 从业人员信息Service业务层处理
 *
 * @author zehong
 * @date 2023-08-16
 */
@Service
public class TPractitionerInfoServiceImpl implements ITPractitionerInfoService
{

    private static final Logger log = LoggerFactory.getLogger(TPractitionerInfoServiceImpl.class);

    @Autowired
    private TPractitionerInfoMapper tPractitionerInfoMapper;

    @Resource
    private TGasStorageStationInfoMapper tGasStorageStationInfoMapper;

    @Resource
    private SysPostMapper sysPostMapper;

    /**
     * 查询从业人员信息
     *
     * @param practitionerId 从业人员信息ID
     * @return 从业人员信息
     */
    @Override
    public TPractitionerInfo selectTPractitionerInfoById(Long practitionerId)
    {
        return tPractitionerInfoMapper.selectTPractitionerInfoById(practitionerId);
    }

    /**
     * 查询从业人员信息列表
     *
     * @param tPractitionerInfo 从业人员信息
     * @return 从业人员信息
     */
    @Override
    public List<TPractitionerInfo> selectTPractitionerInfoList(TPractitionerInfo tPractitionerInfo)
    {
        return tPractitionerInfoMapper.selectTPractitionerInfoList(tPractitionerInfo);
    }

    /**
     * 新增从业人员信息
     *
     * @param tPractitionerInfo 从业人员信息
     * @return 结果
     */
    @Override
    public int insertTPractitionerInfo(TPractitionerInfo tPractitionerInfo)
    {
        tPractitionerInfo.setCreateTime(DateUtils.getNowDate());
        return tPractitionerInfoMapper.insertTPractitionerInfo(tPractitionerInfo);
    }

    /**
     * 修改从业人员信息
     *
     * @param tPractitionerInfo 从业人员信息
     * @return 结果
     */
    @Override
    public int updateTPractitionerInfo(TPractitionerInfo tPractitionerInfo)
    {
        tPractitionerInfo.setUpdateTime(DateUtils.getNowDate());
        return tPractitionerInfoMapper.updateTPractitionerInfo(tPractitionerInfo);
    }

    /**
     * 批量删除从业人员信息
     *
     * @param practitionerIds 需要删除的从业人员信息ID
     * @return 结果
     */
    @Override
    public int deleteTPractitionerInfoByIds(Long[] practitionerIds)
    {
        return tPractitionerInfoMapper.deleteTPractitionerInfoByIds(practitionerIds);
    }

    /**
     * 删除从业人员信息信息
     *
     * @param practitionerId 从业人员信息ID
     * @return 结果
     */
    @Override
    public int deleteTPractitionerInfoById(Long practitionerId)
    {
        return tPractitionerInfoMapper.deleteTPractitionerInfoById(practitionerId);
    }

    /**
     * 从业人员数据导入
     * @param tPractitionerInfoList 从业人员实体
     * @param isUpdateSupport 是否更新
     * @return
     */
    @Override
    public String importTPractitionerInfo(List<TPractitionerInfo> tPractitionerInfoList, Boolean isUpdateSupport){
        if (StringUtils.isNull(tPractitionerInfoList) || tPractitionerInfoList.size() == 0){
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TPractitionerInfo tPractitionerInfo : tPractitionerInfoList){
            try {
                //查询储配站信息
                TGasStorageStationInfo queryGasStorageStationInfo = new TGasStorageStationInfo();
                queryGasStorageStationInfo.setStationName(tPractitionerInfo.getStationName());
                //储配站
                List<TGasStorageStationInfo> gasStorageStationInfo = tGasStorageStationInfoMapper.selectTGasStorageStationInfoList(queryGasStorageStationInfo);
                if(CollectionUtils.isEmpty(gasStorageStationInfo)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、从业人员 " + tPractitionerInfo.getName() +"、储配站" + tPractitionerInfo.getStationName() + " 不存在请创建或导入");
                    continue;
                }

                //岗位
                SysPost post = new SysPost();
                post.setPostName(tPractitionerInfo.getPostName());
                List<SysPost> postInfo = sysPostMapper.selectPostList(post);
                if(CollectionUtils.isEmpty(postInfo)){
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、从业人员 " + tPractitionerInfo.getName() +"、岗位" + tPractitionerInfo.getPostName() + " 不存在请创建");
                    continue;
                }
                TPractitionerInfo query = new TPractitionerInfo();
                query.setPractitionerNum(tPractitionerInfo.getPractitionerNum());
                List<TPractitionerInfo> queryPractitionerInfo = tPractitionerInfoMapper.selectTPractitionerInfoList(query);
                if (StringUtils.isNull(queryPractitionerInfo) || queryPractitionerInfo.size() == 0){
                    tPractitionerInfo.setStationId(gasStorageStationInfo.get(0).getStationId());
                    tPractitionerInfo.setPostId(postInfo.get(0).getPostId());
                    this.insertTPractitionerInfo(tPractitionerInfo);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、工号 " + tPractitionerInfo.getPractitionerNum() + " 导入成功");
                }else if (isUpdateSupport){
                    queryPractitionerInfo.stream().forEach(practitionerInfo ->{
                        tPractitionerInfo.setPractitionerId(practitionerInfo.getPractitionerId());
                        tPractitionerInfo.setStationId(gasStorageStationInfo.get(0).getStationId());
                        tPractitionerInfo.setPostId(postInfo.get(0).getPostId());
                        this.updateTPractitionerInfo(tPractitionerInfo);
                    });
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、工号 " + tPractitionerInfo.getPractitionerNum() + " 更新成功");
                }else{
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、工号 " + tPractitionerInfo.getPractitionerNum() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、工号 " + tPractitionerInfo.getPractitionerNum() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }

    /**
     * 查询登录信息
     * @param tPractitionerInfo
     * @return
     */
    @Override
    public List<TPractitionerInfo> loginTPractitioner(TPractitionerInfo tPractitionerInfo) {
        return tPractitionerInfoMapper.loginTPractitioner(tPractitionerInfo);
    }
}
