package com.zehong.system.task;

import com.serotonin.modbus4j.ModbusMaster;
import com.zehong.system.domain.PalletDeviceBinding;
import com.zehong.system.domain.TEquipmentAlarmData;
import com.zehong.system.domain.TStoreyInfo;
import com.zehong.system.domain.vo.WriteCurrentTimeVo;
import com.zehong.system.mapper.PalletDeviceBindingMapper;
import com.zehong.system.mapper.TStoreyInfoMapper;
import com.zehong.system.modbus.util.Modbus4jUtils;
import com.zehong.system.modbus.util.WriteCurrentTimeUtil;
import com.zehong.system.service.ITEquipmentAlarmDataService;
import org.apache.commons.lang3.StringUtils;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 设备通信Job（修复版：显式暴露错误，避免Trigger变ERROR）
 */
@Component
@DisallowConcurrentExecution // 禁止同一任务并行执行（必须保留）
public class AgingStageOneProcessJob implements Job {
    private static final Logger log = LoggerFactory.getLogger(AgingStageOneProcessJob.class);

    // -------------------------- 常量配置（统一管理，避免魔法值）--------------------------
    // 超时控制：必须小于Cron周期（假设Cron为5分钟，这里留1分钟缓冲）
    private static final int TOTAL_TASK_TIMEOUT_SEC = 240; // 任务总超时：4分钟
    private static final int SINGLE_DEVICE_TIMEOUT_SEC = 10; // 单个设备超时：10秒

    // 全局线程池 - 避免重复创建
    private static final ExecutorService GLOBAL_DEVICE_EXECUTOR = new ThreadPoolExecutor(
            50, 100, 60, TimeUnit.SECONDS,
            new LinkedBlockingQueue<>(500),
            r -> new Thread(r, "global-modbus-device"),
            new ThreadPoolExecutor.CallerRunsPolicy()
    );
    // -------------------------- 依赖注入 --------------------------
    @Resource
    private ITEquipmentAlarmDataService alarmDataService;
    @Resource
    private TStoreyInfoMapper tStoreyInfoMapper;
    @Resource
    private PalletDeviceBindingMapper palletDeviceBindingMapper;

    // -------------------------- 核心执行逻辑 --------------------------
    @Override
    public void execute(JobExecutionContext context) {
        String storeyIdStr = getStoreyIdFromContext(context);
        long startTime = System.currentTimeMillis();

        try {
            TStoreyInfo storeyInfo = validateAndGetStoreyInfo(storeyIdStr);

            // 并行处理3个端口
            List<CompletableFuture<Void>> portFutures = Arrays.asList(
                    processPort(storeyInfo, 501, Arrays.asList(1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27)),
                    processPort(storeyInfo, 502, Arrays.asList(28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54)),
                    processPort(storeyInfo, 503, Arrays.asList(55,56,57,58,59,60,61,62,63,64,65,66,67,68,69,70,71,72))
            );

            // 等待所有端口完成，带总超时
            CompletableFuture<Void> allPorts = CompletableFuture.allOf(
                    portFutures.toArray(new CompletableFuture[0])
            );

            allPorts.get(TOTAL_TASK_TIMEOUT_SEC, TimeUnit.SECONDS);

            log.info("DeviceCommunicationJob任务执行成功: fStoreyId={}, 耗时={}ms",
                    storeyIdStr, System.currentTimeMillis() - startTime);

        } catch (TimeoutException e) {
            log.warn("任务执行超时: fStoreyId={}", storeyIdStr);
            recordAlarm(null, storeyIdStr, "任务执行超时");
        } catch (Exception e) {
            log.error("任务执行异常: fStoreyId={}", storeyIdStr, e);
            recordAlarm(null, storeyIdStr, "任务执行异常: " + e.getMessage());
        }
    }
    /**
     * 处理单个端口的所有设备
     */
    private CompletableFuture<Void> processPort(TStoreyInfo storeyInfo, int port, List<Integer> deviceIds) {
        return CompletableFuture.runAsync(() -> {
            String ip = storeyInfo.getfIp();
            String storeyIdStr = storeyInfo.getfStoreyId().toString();

            log.info("开始端口通信: ip={}, port={}, 设备数={}", ip, port, deviceIds.size());

            AtomicInteger errorCount = new AtomicInteger(0);

            // 并行处理该端口的所有设备
            List<CompletableFuture<Boolean>> deviceFutures = deviceIds.stream()
                    .map(deviceId -> processDeviceWithWrite(ip, port, deviceId, errorCount))
                    .collect(Collectors.toList());

            try {
                // 等待该端口所有设备完成
                CompletableFuture<Void> allDevices = CompletableFuture.allOf(
                        deviceFutures.toArray(new CompletableFuture[0])
                );

                // 端口超时 = 设备数 * 单设备超时 / 并发因子
                int portTimeout = Math.max(30, deviceIds.size() * SINGLE_DEVICE_TIMEOUT_SEC / 5);
                allDevices.get(portTimeout, TimeUnit.SECONDS);

            } catch (TimeoutException e) {
                log.warn("端口{}通信超时: ip={}, fStoreyId={}", port, ip, storeyIdStr);
                recordAlarm(storeyInfo, "端口" + port + "通信超时");
            } catch (Exception e) {
                log.error("端口{}通信异常: ip={}, fStoreyId={}", port, ip, storeyIdStr, e);
            }

            if (errorCount.get() > 0) {
                log.warn("端口{}部分设备失败: 失败数={}, fStoreyId={}",
                        port, errorCount.get(), storeyIdStr);
            }

            log.info("端口通信完成: ip={}, port={}, fStoreyId={}", ip, port, storeyIdStr);
        }, GLOBAL_DEVICE_EXECUTOR);
    }

    /**
     * 处理单个设备（读取 + 条件写入）
     */
    private CompletableFuture<Boolean> processDeviceWithWrite(String ip, int port, int deviceId, AtomicInteger errorCount) {
        return CompletableFuture.supplyAsync(() -> {
            ModbusMaster master = null;
            PalletDeviceBinding binding = null;
            try {
                // 1. 读取设备数据
                int[] result = Modbus4jUtils.readDeviceWithRetry(ip, port, deviceId);

                // 2. 查询设备绑定信息
                binding = palletDeviceBindingMapper.selectByTrayIdAndIndex(ip, deviceId);
                if (binding == null) {
                    log.warn("未找到设备绑定: ip={}, deviceId={}", ip, deviceId);
                    recordAlarm(null, "ip:" + ip + ",port:" + port + ",deviceId:" + deviceId, "未找到设备绑定");
                    errorCount.incrementAndGet();
                    return false;
                }

                // 3. 更新设备状态
                binding.setStatus(String.valueOf(result[1]));

                // 3. 更新浓度值
                binding.setConcentration(String.valueOf(result[0]));

                // 重用之前的master连接进行写操作
                master = Modbus4jUtils.createModbusMaster(ip, port);
                // 4. 条件写入时间
                if (result[1] == 1 || result[1] == 3 || result[1] == 4) {
//                    writeCurrentTimeToDevice(master, deviceId, binding);
                    WriteCurrentTimeVo writeCurrentTimeVo = WriteCurrentTimeUtil.generateWriteCurrentTimeVo();
                    boolean allSuccess = Modbus4jUtils.writeCurrentTimeToDeviceEnhanced(master, deviceId,writeCurrentTimeVo);
                    if (!allSuccess) {
                        log.warn("设备{}写入时间失败: ip={}, port={}", deviceId, ip, port);
                        binding.setWriteTimeStatus("0");
                    } else {
                        binding.setRecordYear(String.valueOf(writeCurrentTimeVo.getYear()));
                        binding.setRecordMonth(String.valueOf(writeCurrentTimeVo.getMonth()));
                        binding.setRecordDate(String.valueOf(writeCurrentTimeVo.getDay()));
                        binding.setRecordHour(String.valueOf(writeCurrentTimeVo.getHour()));
                        binding.setRecordMinute(String.valueOf(writeCurrentTimeVo.getMinute()));
                        binding.setWriteTimeStatus("1");
                    }

                    // 5. 写入自检让设备开始自检，跟 上面的状态没关系
                    // 20251231  写自检 根据状态判断
                    int i = Modbus4jUtils.writeSelfCheckStatus(master, deviceId);
                    binding.setWriteSelfCheckStatus(i);
                }
                // 6. 更新数据库
                palletDeviceBindingMapper.updatePalletDeviceBinding(binding);

                log.info("设备{}处理完成: ip={}, port={}, status={}", deviceId, ip, port, result[1]);
                if((result[1] == 1 || result[1] == 3 || result[1] == 4) && "0".equals(binding.getWriteTimeStatus())) {
                    errorCount.incrementAndGet();
                    return false;
                }
                return true;

            } catch (Exception e) {
                log.info("设备{}处理异常: ip={}, port={}", deviceId, ip, port, e);
                if(binding != null) {
                    binding.setStatus("5");
                    binding.setWriteTimeStatus("0");
                    palletDeviceBindingMapper.updatePalletDeviceBinding(binding);
                }
                errorCount.incrementAndGet();
                return false;
            } finally {
                Modbus4jUtils.destroyModbusMaster(master, deviceId);
            }
        }, GLOBAL_DEVICE_EXECUTOR);
    }


    /**
     * 验证参数并获取设备信息（参数错误直接抛出异常）
     */
    private TStoreyInfo validateAndGetStoreyInfo(String storeyIdStr) {
        if (StringUtils.isBlank(storeyIdStr)) {
            log.error("fStoreyId参数为空");
            return null;
        }

        try {
            Long storeyId = Long.parseLong(storeyIdStr);
            TStoreyInfo storeyInfo = tStoreyInfoMapper.selectTStoreyInfoById(storeyId);

            if (storeyInfo == null || StringUtils.isBlank(storeyInfo.getfIp())) {
                log.error("设备信息无效: fStoreyId={}", storeyIdStr);
                return null;
            }

            return storeyInfo;
        } catch (NumberFormatException e) {
            log.error("fStoreyId格式错误: {}", storeyIdStr);
            return null;
        }
    }

    /**
     * 写入当前时间到设备
     */
    private void writeCurrentTimeToDevice(ModbusMaster master, int deviceId,
                                          PalletDeviceBinding binding) {
        try {
            Calendar cal = Calendar.getInstance();
            int year = cal.get(Calendar.YEAR);
            int month = cal.get(Calendar.MONTH) + 1;
            int day = cal.get(Calendar.DATE);
            int hour = cal.get(Calendar.HOUR_OF_DAY);
            int minute = cal.get(Calendar.MINUTE);

            // 写入时间寄存器
            boolean success;
            success = Modbus4jUtils.writeRegister(master, deviceId, 4, (short) year);
            success &= Modbus4jUtils.writeRegister(master, deviceId, 5, (short) month);
            success &= Modbus4jUtils.writeRegister(master, deviceId, 6, (short) day);
            success &= Modbus4jUtils.writeRegister(master, deviceId, 7, (short) hour);
            success &= Modbus4jUtils.writeRegister(master, deviceId, 8, (short) minute);

            if (success) {
                int[] ints = Modbus4jUtils.readDeviceRegisters(master, deviceId);
                if(ints[4] == year && ints[5] == month && ints[6] == day && ints[7] == hour && ints[8] == minute) {
                    binding.setRecordYear(String.valueOf(year));
                    binding.setRecordMonth(String.valueOf(month));
                    binding.setRecordDate(String.valueOf(day));
                    binding.setRecordHour(String.valueOf(hour));
                    binding.setRecordMinute(String.valueOf(minute));
                    binding.setWriteTimeStatus("1");
                    log.debug("设备{}时间写入成功", deviceId);
                } else {
                    log.info("设备{}时间写入失败: year={}, month={}, day={}, hour={}, minute={}",deviceId,ints[4], ints[5], ints[6], ints[7], ints[8]);
                    binding.setWriteTimeStatus("0");
                    recordAlarmByBinding(binding, "设备时间写入失败");
                }
            } else {
                binding.setWriteTimeStatus("0");
                recordAlarmByBinding(binding, "设备时间写入失败");
            }
        } catch (Exception e) {
            binding.setWriteTimeStatus("0");
            log.error("设备{}时间写入异常", deviceId, e);
            recordAlarmByBinding(binding, "设备时间写入异常: " + e.getMessage());
        }
    }


    // -------------------------- 辅助方法（日志/告警）--------------------------
    /**
     * 从JobContext中获取fStoreyId（失败返回unknown）
     */
    private String getStoreyIdFromContext(JobExecutionContext context) {
        try {
            JobDataMap data = context.getJobDetail().getJobDataMap();
            return data != null ? data.getString("fStoreyId") : "unknown";
        } catch (Exception e) {
            log.error("从JobContext获取fStoreyId失败", e);
            return "unknown";
        }
    }

    /**
     * 记录告警（兼容设备信息为空的场景）
     */
    private void recordAlarm(TStoreyInfo storeyInfo, String equipmentCode, String alarmData) {
        try {
            TEquipmentAlarmData alarm = new TEquipmentAlarmData();
            alarm.setfAlarmType("03"); // 老化层告警
            alarm.setfEquipmentCode(storeyInfo != null ? storeyInfo.getfStoreyCode() : equipmentCode);
            alarm.setfAlarmData(alarmData);
            alarm.setfCreateTime(new Date());
            alarmDataService.insertTEquipmentAlarmData(alarm);
            log.debug("告警记录成功：设备编码={}，内容={}", alarm.getfEquipmentCode(), alarmData);
        } catch (Exception e) {
            log.error("告警记录失败：设备编码={}，内容={}", equipmentCode, alarmData, e);
        }
    }

    /**
     * 简化告警记录（设备信息非空时）
     */
    private void recordAlarm(TStoreyInfo storeyInfo, String alarmData) {
        recordAlarm(storeyInfo, storeyInfo != null ? storeyInfo.getfStoreyCode() : "unknown", alarmData);
    }


    /**
     * 统一告警记录（修复字段错误，确保写入成功）
     */
    private void recordAlarmByBinding(PalletDeviceBinding binding, String alarmMsg) {
        String equipmentCode = binding != null ? binding.getDeviceCode() : "unknown";
        recordAlarmByBingding(equipmentCode, alarmMsg);
    }

    private void recordAlarmByBingding(String equipmentCode, String alarmMsg) {
        try {
            TEquipmentAlarmData alarm = new TEquipmentAlarmData();
            alarm.setfAlarmType("04"); // 04.点位告警
            alarm.setfEquipmentCode(equipmentCode);
            alarm.setfAlarmData(alarmMsg);
            alarm.setfCreateTime(new Date()); // 修复字段错误：用fCreateTime而非createTime
            alarmDataService.insertTEquipmentAlarmData(alarm);
            log.debug("告警记录成功：equipmentCode={}, msg={}", equipmentCode, alarmMsg);
        } catch (Exception e) {
            log.error("告警记录失败：equipmentCode={}, msg={}", equipmentCode, alarmMsg, e);
        }
    }
}
