package com.zehong.system.task;

import com.serotonin.modbus4j.ModbusMaster;
import com.serotonin.modbus4j.exception.ErrorResponseException;
import com.serotonin.modbus4j.exception.ModbusInitException;
import com.serotonin.modbus4j.exception.ModbusTransportException;
import com.zehong.system.domain.TEquipmentAlarmData;
import com.zehong.system.domain.TStoreyInfo;
import com.zehong.system.modbus.util.Modbus4jUtils;
import com.zehong.system.service.ITEquipmentAlarmDataService;
import com.zehong.system.service.ITStoreyInfoService;
import org.quartz.SchedulerException;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;

/**
 * @author lenovo
 * @date 2025/6/17
 * @description 事件接收器
 */
@Component
public class AllCommandHandler {
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(AllCommandHandler.class);

    @Resource
    private ITEquipmentAlarmDataService alarmDataService;
    @Resource
    private ITStoreyInfoService tStoreyInfoService;

    @Resource
    private DeviceTaskScheduler deviceTaskScheduler;

    /**
     * check是否启动 ，没启动就启动下 开始老化
     * @param event event
     */
    @Async
    @EventListener(CheckPowerOnCommandEvent.class)
    public void handleCheckPowerOnCommand(CheckPowerOnCommandEvent event) {

        String fip = event.getIp();
        int fport = event.getPort();
        int registerOffset = event.getRegisterOffset();
        int layer = event.getLayer();

        String storeyCode = event.getDeviceCode() + "-" + (event.getLayer()+1);

        log.info("需要发送上电指令 - 设备:{} 层:{} ip:{} 端口号:{}", event.getDeviceCode(),layer, fip, fport);
        ModbusMaster master;
        try {
            master = Modbus4jUtils.getMaster(fip, fport);
            Boolean aBoolean = Modbus4jUtils.readCoilStatus(master, 1, registerOffset);

            log.info("当前 - 设备:{} 层:{} -的 状态是:{}", event.getDeviceCode(), event.getLayer(),aBoolean);
            if(!aBoolean) {
                Modbus4jUtils.writeCoil(master, 1, registerOffset, true);

                TStoreyInfo tStoreyInfo = tStoreyInfoService.selectTStoreyInfoByCode(storeyCode);
                tStoreyInfo.setfStatus("1");
                tStoreyInfo.setfAgingStartTime(new Date());
                tStoreyInfoService.updateTStoreyInfo(tStoreyInfo);

                deviceTaskScheduler.scheduleDeviceMonitoring(tStoreyInfo.getfStoreyId(),fip,fport);
            }

        } catch (ModbusInitException | ModbusTransportException | ErrorResponseException | SchedulerException e) {
            // 记录异常
            TEquipmentAlarmData alarmData = new TEquipmentAlarmData();
            alarmData.setfAlarmType("03"); // 老化层
            alarmData.setfEquipmentCode(event.getDeviceCode());
            alarmData.setfAlarmData("上电或者是启动任务检测时异常: " + e.getMessage());
            alarmDataService.insertTEquipmentAlarmData(alarmData);
        }

    }

    @Async  // 异步执行
    @EventListener(PowerOffCommandEvent.class)
    public void handlePowerOffCommand(PowerOffCommandEvent event) {
        try {
            String storeyCode = event.getDeviceCode() + "-" + event.getLayer();
            String ip = event.getIp();
            int port = event.getPort();
            log.info("需要发送断电指令 - 设备:{} 层:{}", event.getDeviceCode(), event.getLayer());
            TStoreyInfo tStoreyInfo = tStoreyInfoService.selectTStoreyInfoByCode(storeyCode);
            if (tStoreyInfo == null) {
                TEquipmentAlarmData alarmData = new TEquipmentAlarmData();
                // 记录异常数据
                alarmData.setfAlarmType("01"); //01.老化柜 02.机械臂 03.老化层 04.点位
                alarmData.setfEquipmentCode(event.getDeviceCode());
                alarmData.setfAlarmData("下属" + storeyCode + "号老化层不存在");
                alarmDataService.insertTEquipmentAlarmData(alarmData);
            } else {
                    //
                    ModbusMaster master = Modbus4jUtils.getMaster(ip, port);
                    Modbus4jUtils.writeCoil(master, 1, event.getLayer(), false);
                    log.info("已发送断电指令 - 设备:{} 层:{}", event.getDeviceCode(), event.getLayer());
                    master.destroy();
            }
        } catch (ModbusInitException | ModbusTransportException e) {
            log.error("断电指令执行失败 - 设备:{} 层:{}", event.getDeviceCode(), event.getLayer(), e);

            // 记录异常
            TEquipmentAlarmData alarmData = new TEquipmentAlarmData();
            alarmData.setfAlarmType("03"); // 老化层
            alarmData.setfEquipmentCode(event.getDeviceCode());
            alarmData.setfAlarmData("断电指令执行失败: " + e.getMessage());
            alarmDataService.insertTEquipmentAlarmData(alarmData);
        }
    }
}
