package com.zehong.system.service.impl;

import java.util.*;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import com.zehong.common.constant.RoboticArmConstans;
import com.zehong.system.netty.handler.NettyUdpServerHandler;
import org.springframework.stereotype.Service;
import com.zehong.common.annotation.DataSource;
import com.zehong.common.constant.Constants;
import com.zehong.common.constant.UserConstants;
import com.zehong.common.core.redis.RedisCache;
import com.zehong.common.core.text.Convert;
import com.zehong.common.enums.DataSourceType;
import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.SysConfig;
import com.zehong.system.mapper.SysConfigMapper;
import com.zehong.system.service.ISysConfigService;

/**
 * 参数配置 服务层实现
 * 
 * @author zehong
 */
@Service
public class SysConfigServiceImpl implements ISysConfigService
{
    @Resource
    private SysConfigMapper configMapper;

    @Resource
    private RedisCache redisCache;

    /**
     * 项目启动时，初始化参数到缓存
     */
    @PostConstruct
    public void init()
    {
        loadingConfigCache();
    }

    /**
     * 查询参数配置信息
     * 
     * @param configId 参数配置ID
     * @return 参数配置信息
     */
    @Override
    @DataSource(DataSourceType.MASTER)
    public SysConfig selectConfigById(Long configId)
    {
        SysConfig config = new SysConfig();
        config.setConfigId(configId);
        return configMapper.selectConfig(config);
    }

    /**
     * 根据键名查询参数配置信息
     * 
     * @param configKey 参数key
     * @return 参数键值
     */
    @Override
    public String selectConfigByKey(String configKey)
    {
        String configValue = Convert.toStr(redisCache.getCacheObject(getCacheKey(configKey)));
        if (StringUtils.isNotEmpty(configValue))
        {
            return configValue;
        }
        SysConfig config = new SysConfig();
        config.setConfigKey(configKey);
        SysConfig retConfig = configMapper.selectConfig(config);
        if (StringUtils.isNotNull(retConfig))
        {
            redisCache.setCacheObject(getCacheKey(configKey), retConfig.getConfigValue());
            return retConfig.getConfigValue();
        }
        return StringUtils.EMPTY;
    }

    /**
     * 根据键名查询参数配置信息-直接查询不经过redis
     *
     * @param configKey 键名
     * @return 参数键值
     */
    @Override
    public String directSelectConfigByKey(String configKey) {
        if(StringUtils.isBlank(configKey)) {
            return StringUtils.EMPTY;
        }
        SysConfig config = new SysConfig();
        config.setConfigKey(configKey);
        SysConfig sysConfig = configMapper.selectConfig(config);
        if(sysConfig != null) {
            return sysConfig.getConfigValue();
        }
        return StringUtils.EMPTY;
    }

    /**
     * 查询 agingStage1Time agingStage2Time agingStage3Time
     *
     * @return r
     */
    @Override
    public List<SysConfig> selectAgingStageTime() {

        List<SysConfig> sysConfigs = configMapper.selectAgingStageTime();
        if(sysConfigs != null && sysConfigs.size() > 0) {
            return sysConfigs;
        }

        return new ArrayList<>();
    }

    @Override
    public SysConfig selectAgingStage3Time() {

        return configMapper.checkConfigKeyUnique(RoboticArmConstans.AGING_STAGE3_TIME);
    }

    /**
     * 更新 agingStage1Time agingStage2Time agingStage3Time
     *
     * @param sysConfigs s
     * @return r
     */
    @Override
    public int updateAgingStageTime(List<SysConfig> sysConfigs) {

        for(SysConfig sysConfig : sysConfigs) {
            configMapper.directUpdateConfigByKey(sysConfig);
        }

        return 1;
    }

    /**
     * 查询参数配置列表
     * 
     * @param config 参数配置信息
     * @return 参数配置集合
     */
    @Override
    public List<SysConfig> selectConfigList(SysConfig config)
    {
        return configMapper.selectConfigList(config);
    }

    /**
     * 新增参数配置
     * 
     * @param config 参数配置信息
     * @return 结果
     */
    @Override
    public int insertConfig(SysConfig config)
    {
        int row = configMapper.insertConfig(config);
        if (row > 0)
        {
            redisCache.setCacheObject(getCacheKey(config.getConfigKey()), config.getConfigValue());
        }
        return row;
    }

    /**
     * 修改参数配置
     * 
     * @param config 参数配置信息
     * @return 结果
     */
    @Override
    public int updateConfig(SysConfig config)
    {
        int row = configMapper.updateConfig(config);
        if (row > 0)
        {
            redisCache.setCacheObject(getCacheKey(config.getConfigKey()), config.getConfigValue());
        }
        return row;
    }

    /**
     * 修改上传地址
     *
     * @return 结果
     */
    @Override
    public int updateUploadMesAddress(SysConfig sysConfig) {
        sysConfig.setConfigKey(RoboticArmConstans.UPLOAD_MES_ADDRESS);
        return configMapper.directUpdateConfigByKey(sysConfig);
    }

    /**
     * 批量删除参数信息
     * 
     * @param configIds 需要删除的参数ID
     * @return 结果
     */
    @Override
    public void deleteConfigByIds(Long[] configIds)
    {
        for (Long configId : configIds)
        {
            SysConfig config = selectConfigById(configId);
            if (StringUtils.equals(UserConstants.YES, config.getConfigType()))
            {
                throw new CustomException(String.format("内置参数【%1$s】不能删除 ", config.getConfigKey()));
            }
            configMapper.deleteConfigById(configId);
            redisCache.deleteObject(getCacheKey(config.getConfigKey()));
        }
    }

    /**
     * 加载参数缓存数据
     */
    @Override
    public void loadingConfigCache()
    {
        List<SysConfig> configsList = configMapper.selectConfigList(new SysConfig());
        for (SysConfig config : configsList)
        {
            redisCache.setCacheObject(getCacheKey(config.getConfigKey()), config.getConfigValue());
        }
    }

    /**
     * 清空参数缓存数据
     */
    @Override
    public void clearConfigCache()
    {
        Collection<String> keys = redisCache.keys(Constants.SYS_CONFIG_KEY + "*");
        redisCache.deleteObject(keys);
    }

    /**
     * 重置参数缓存数据
     */
    @Override
    public void resetConfigCache()
    {
        clearConfigCache();
        loadingConfigCache();
    }

    /**
     * 校验参数键名是否唯一
     * 
     * @param config 参数配置信息
     * @return 结果
     */
    @Override
    public String checkConfigKeyUnique(SysConfig config)
    {
        Long configId = StringUtils.isNull(config.getConfigId()) ? -1L : config.getConfigId();
        SysConfig info = configMapper.checkConfigKeyUnique(config.getConfigKey());
        if (StringUtils.isNotNull(info) && info.getConfigId().longValue() != configId.longValue())
        {
            return UserConstants.NOT_UNIQUE;
        }
        return UserConstants.UNIQUE;
    }

    /**
     * 设置cache key
     * 
     * @param configKey 参数键
     * @return 缓存键key
     */
    private String getCacheKey(String configKey)
    {
        return Constants.SYS_CONFIG_KEY + configKey;
    }
}
