package com.zehong.system.service.impl;

import java.util.List;

import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.SecurityUtils;
import com.zehong.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.zehong.system.mapper.TDeviceInfoMapper;
import com.zehong.system.domain.TDeviceInfo;
import com.zehong.system.service.ITDeviceInfoService;

/**
 * 设备信息管理Service业务层处理
 * 
 * @author zehong
 * @date 2022-07-02
 */
@Service
public class TDeviceInfoServiceImpl implements ITDeviceInfoService
{
    private static final Logger log = LoggerFactory.getLogger(TDeviceInfoServiceImpl.class);

    @Autowired
    private TDeviceInfoMapper tDeviceInfoMapper;

    /**
     * 查询设备信息管理
     * 
     * @param id 设备信息管理ID
     * @return 设备信息管理
     */
    @Override
    public TDeviceInfo selectTDeviceInfoById(Long id)
    {
        return tDeviceInfoMapper.selectTDeviceInfoById(id);
    }

    /**
     * 查询设备信息管理列表
     * 
     * @param tDeviceInfo 设备信息管理
     * @return 设备信息管理
     */
    @Override
    public List<TDeviceInfo> selectTDeviceInfoList(TDeviceInfo tDeviceInfo)
    {
        return tDeviceInfoMapper.selectTDeviceInfoList(tDeviceInfo);
    }

    /**
     * 新增设备信息管理
     * 
     * @param tDeviceInfo 设备信息管理
     * @return 结果
     */
    @Override
    public int insertTDeviceInfo(TDeviceInfo tDeviceInfo)
    {
        tDeviceInfo.setCreateTime(DateUtils.getNowDate());
        return tDeviceInfoMapper.insertTDeviceInfo(tDeviceInfo);
    }

    /**
     * 修改设备信息管理
     * 
     * @param tDeviceInfo 设备信息管理
     * @return 结果
     */
    @Override
    public int updateTDeviceInfo(TDeviceInfo tDeviceInfo)
    {
        tDeviceInfo.setUpdateTime(DateUtils.getNowDate());
        return tDeviceInfoMapper.updateTDeviceInfo(tDeviceInfo);
    }

    /**
     * 批量删除设备信息管理
     * 
     * @param ids 需要删除的设备信息管理ID
     * @return 结果
     */
    @Override
    public int deleteTDeviceInfoByIds(Long[] ids)
    {
        return tDeviceInfoMapper.deleteTDeviceInfoByIds(ids);
    }

    /**
     * 删除设备信息管理信息
     * 
     * @param id 设备信息管理ID
     * @return 结果
     */
    @Override
    public int deleteTDeviceInfoById(Long id)
    {
        return tDeviceInfoMapper.deleteTDeviceInfoById(id);
    }

    /**
     * 导入设备数据
     *
     * @param TDeviceInfo 设备数据列表
     * @param isUpdateSupport 是否更新支持，如果已存在，则进行更新数据
     * @param
     * @return 结果
     */
    @Override
    public String importDevice(List<TDeviceInfo> TDeviceInfo, Boolean isUpdateSupport)
    {
        if (StringUtils.isNull(TDeviceInfo) || TDeviceInfo.size() == 0)
        {
            throw new CustomException("导入设备数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TDeviceInfo device : TDeviceInfo)
        {
            try
            {
                // 验证是否存在这个用户
                TDeviceInfo d = new TDeviceInfo();
                d.setDeviceCode(device.getDeviceCode());
                List<TDeviceInfo> list = tDeviceInfoMapper.selectTDeviceInfoList(d);
                if (list.size() == 0)
                {
                    this.insertTDeviceInfo(device);
                    successNum++;
//                    successMsg.append("<br/>" + successNum + "、设备 " + device.getDeviceName() + "(" + device.getDeviceCode() + ")" + " 导入成功");
                }
                else if (isUpdateSupport)
                {
                    this.updateTDeviceInfo(device);
                    successNum++;
//                    successMsg.append("<br/>" + successNum + "、设备 " + device.getDeviceName() + "(" + device.getDeviceCode() + ")" + " 更新成功");
                }
                else
                {
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、设备编号 " + device.getDeviceCode() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、设备 " + device.getDeviceName() + "(" + device.getDeviceCode() + ")" + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "导入完成，共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "数据已全部导入成功！共 " + successNum + " 条");
        }
        return successMsg.toString();
    }
}
