package com.zehong.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.zehong.constant.MesDeviceDataConstant;
import com.zehong.constant.PCBACodeRules;
import com.zehong.dao.PCBADevicesMapper;
import com.zehong.entity.PCBADevices;
import com.zehong.service.ScanCodeResultService;
import com.zehong.utils.RedisCache;
import com.zehong.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 扫码结果
 */
@Service
@DS("mes")
public class ScanCodeResultServiceImpl implements ScanCodeResultService{

    private static final Logger log = LoggerFactory.getLogger(ScanCodeResultServiceImpl.class);

    @Autowired
    private RedisCache redisCache;

    @Resource
    private PCBADevicesMapper pcbaDevicesMapper;

    /**
     * 扫码规则
     */
    private final Map<String,Pattern> rules = new HashMap<>();

    {
        for(PCBACodeRules rule: PCBACodeRules.values()){
            rules.put(rule.name(),Pattern.compile(rule.getRegex()));
        }
    }

    /**
     * 扫码结果
     * @param message 扫码信息
     * @return int
     */
    @Override
    public int scanCodeResult(String message) {
        log.info("扫码结果字符串=========" + message);
        //解析字符串
        Map<String,Object> analysisResult = analysis(message);
        //数据处理
        return dealStrategy(analysisResult);
    }

    /**
     * 解析扫码结果
     * @param message 扫码信息
     */
    private Map<String,Object> analysis(String message){
        Map<String,Object> analysisResult = new HashMap<>();
        for(Map.Entry<String, Pattern> rule : rules.entrySet()){
            List<String> codes = new ArrayList<>();
            Matcher m = rule.getValue().matcher(message);
            while (m.find()){
                String code = m.group();
                codes.add(code);
                message = message.replace(code,"");
            }
            if(!CollectionUtils.isEmpty(codes)) analysisResult.put(rule.getKey(),codes);
        }
        //nb码无规则
        if(StringUtils.isNotEmpty(message)) analysisResult.put("nbCode",getNbCode(message));
        return analysisResult;
    }

    /**
     * 处理策略
     * @param analysisResult 解析结果
     */
    private int dealStrategy(Map<String,Object> analysisResult){
        // NB码 + 卡批次码
        if(analysisResult.containsKey("nbCode") && analysisResult.containsKey(PCBACodeRules.CARDBATCHCODE.name())){
            return addNbAndCardCode(analysisResult);
        }
        // 主板码 + 传感器批次码
        if(analysisResult.containsKey(PCBACodeRules.MAINBOARDCODE.name()) && analysisResult.containsKey(PCBACodeRules.SENSORBATCHCODE.name())){
            return addMainBoardCodeAndSensorBatchCode(analysisResult);
        }
        //主板码 + NB码
        if(analysisResult.containsKey(PCBACodeRules.MAINBOARDCODE.name()) && analysisResult.containsKey("nbCode")){
            return updateNbCodeByMainBoardCode(analysisResult);
        }
        return 0;
    }

    /**
     * 根据不同厂家获取nb码
     * @param message nb码字符串
     * @return String
     */
    private String getNbCode(String message){
        if(message.contains("_")){
            return message.split("_")[0];
        }
        if (message.contains(";")){
            return message.split(";")[0];
        }
        return message;
    }


    /**
     * 新增nb+卡批次码
     * @param analysisResult 解析结果
     * @return int
     */
    private int addNbAndCardCode(Map<String,Object> analysisResult){
        Map<String,Object> cache = redisCache.getCacheMap(MesDeviceDataConstant.EQUIPMENT_DOCKING + ":" + MesDeviceDataConstant.PCBA_DEVICE_KEY);
        if(null != cache && cache.containsKey("pcbaProductionTasksNumber") && cache.containsKey("materialInfoId")){
            PCBADevices devices = new PCBADevices();
            devices.setMotherboardCode(analysisResult.get("nbCode").toString());
            List<String> codes = (List<String>) analysisResult.get(PCBACodeRules.CARDBATCHCODE.name());
            devices.setIotCardBatchNumber(codes.get(0));
            devices.setPcbaProductionTasksNumber(cache.get("pcbaProductionTasksNumber").toString());
            devices.setMaterialInfoId((Long)cache.get("materialInfoId"));
            devices.setCreateTime(new Date());
            return pcbaDevicesMapper.insertPcbaDevices(devices);
        }
        return 0;
    }

    /**
     * 新增主板码+传感器批次码
     * @param analysisResult 解析结果
     * @return int
     */
    private int addMainBoardCodeAndSensorBatchCode(Map<String,Object> analysisResult){
        Map<String,Object> cache = redisCache.getCacheMap(MesDeviceDataConstant.EQUIPMENT_DOCKING + ":" + MesDeviceDataConstant.PCBA_DEVICE_KEY);
        if(null != cache && cache.containsKey("pcbaProductionTasksNumber") && cache.containsKey("materialInfoId")){
            PCBADevices devices = new PCBADevices();
            List<String> mainCodes = (List<String>) analysisResult.get(PCBACodeRules.MAINBOARDCODE.name());
            devices.setMotherboardCode(mainCodes.get(0));
            List<String> sensorCodes = (List<String>) analysisResult.get(PCBACodeRules.SENSORBATCHCODE.name());
            devices.setSensorCode(sensorCodes.get(0));
            devices.setPcbaProductionTasksNumber(cache.get("pcbaProductionTasksNumber").toString());
            devices.setMaterialInfoId((Long)cache.get("materialInfoId"));
            devices.setCreateTime(new Date());
            return pcbaDevicesMapper.insertPcbaDevices(devices);
        }
        return 0;
    }

    /**
     * 修改主板码和nb码
     * @param analysisResult 解析结果
     * @return int
     */
    private int updateNbCodeByMainBoardCode(Map<String,Object> analysisResult){
        PCBADevices devices = new PCBADevices();
        List<String> mainCodes = (List<String>) analysisResult.get(PCBACodeRules.MAINBOARDCODE.name());

        List<PCBADevices> nbCode = pcbaDevicesMapper.selectPcbaDeviceInfoByNbBoradCodeAndTasksNumber(analysisResult.get("nbCode").toString(), mainCodes.get(0));
        if(CollectionUtils.isEmpty(nbCode)) {
            return 0;
        }
        PCBADevices pcbaDevices = nbCode.get(0);
        String iotCardBatchNumber = pcbaDevices.getIotCardBatchNumber();

        devices.setMotherboardCode(mainCodes.get(0));
        devices.setNbCode(analysisResult.get("nbCode").toString());
        devices.setIotCardBatchNumber(iotCardBatchNumber);
        devices.setCreateTime(new Date());
        return pcbaDevicesMapper.updatePcbaDeviceInfoByMotherBoardCode(devices);
    }
}
