package com.zehong.system.service.impl.shelf.aspectj;

import com.alibaba.fastjson.JSONObject;
import com.zehong.system.domain.shelf.TPlatformDockingTask;
import com.zehong.system.mapper.shelf.TPlatformDockingTaskMapper;
import com.zehong.system.service.impl.shelf.annotation.DockingTaskType;
import com.zehong.system.service.impl.shelf.annotation.PlatformDockingTaskLog;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 平台对接任务消息持久化切面
 */
@Aspect
@Component
public class PlatformDockingTaskAspect {

    private static final Logger log = LoggerFactory.getLogger(PlatformDockingTaskAspect.class);

    @Resource
    private TPlatformDockingTaskMapper platformDockingTaskMapper;

    // 定义切点：拦截带有@PlatformDockingTaskLog注解的方法
    @Pointcut("@annotation(platformDockingTaskLog)")
    public void platformDockingTaskPointcut(PlatformDockingTaskLog platformDockingTaskLog) {}

    // 后置通知：在方法执行后记录日志信息
    @AfterReturning(pointcut="platformDockingTaskPointcut(platformDockingTaskLog)",returning = "result")
    public void insertPlatformDockingTaskLog(JoinPoint joinPoint,PlatformDockingTaskLog platformDockingTaskLog, Object result) {
        DockingTaskType type = platformDockingTaskLog.type();
        if(DockingTaskType.CALL_RACK_SYSTEM.equals(type)){
            insertCallRackSystemLog(joinPoint,result);
        }
        if(DockingTaskType.RECEIVE_RACK_SYSTEM_INFO.equals(type)){
            insertReceiveRackSystemInfo(joinPoint);
        }
    }
    /**
     * 调用料架系统记录
     * @param joinPoint 请求参数
     * @param result 返回结果
     */
    private void insertCallRackSystemLog(JoinPoint joinPoint,Object result){
        String param = (String) joinPoint.getArgs()[0];
        JSONObject paramJson = JSONObject.parseObject(param);
        //判断是否为入库
        if(paramJson.containsKey("inStoreParam")){
            paramJson = paramJson.getJSONObject("inStoreParam");
        }
        //插入日志
        TPlatformDockingTask tPlatformDockingTask = new TPlatformDockingTask();
        tPlatformDockingTask.setfTaskId(paramJson.getString("ID"));
        tPlatformDockingTask.setfOpType(paramJson.getString("OPTYPE"));
        tPlatformDockingTask.setfShelf(paramJson.getString("SHELF"));
        tPlatformDockingTask.setfColor(paramJson.getString("COLOR"));
        tPlatformDockingTask.setfUsername(paramJson.getString("USERNAME"));
        tPlatformDockingTask.setfType(0);
        tPlatformDockingTask.setfTotalInfo(paramJson.getString("TOTALINFO"));
        tPlatformDockingTask.setfRequestTime(new Date());
        tPlatformDockingTask.setfResponseInfo((String) result);
        platformDockingTaskMapper.insertTPlatformDockingTask(tPlatformDockingTask);
    }

    /**
     * 料架系统回复记录
     * @param joinPoint 请求参数
     */
    private void insertReceiveRackSystemInfo(JoinPoint joinPoint){
        try {
            JSONObject request = (JSONObject) joinPoint.getArgs()[0];
            //插入日志
            TPlatformDockingTask tPlatformDockingTask = new TPlatformDockingTask();
            tPlatformDockingTask.setfTaskId(request.getString("id"));
            tPlatformDockingTask.setfOpType(request.getString("optype"));
            tPlatformDockingTask.setfShelf(request.getString("shelf"));
            tPlatformDockingTask.setfLocation(request.getString("location"));
            tPlatformDockingTask.setfRid(request.getString("rid"));
            tPlatformDockingTask.setfType(1);
            tPlatformDockingTask.setfUsername(request.getString("userName"));
            tPlatformDockingTask.setfOperateDate(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse(request.getString("operateDate")));
            platformDockingTaskMapper.insertTPlatformDockingTask(tPlatformDockingTask);
        } catch (ParseException e) {
            log.error("料架系统回复记录持久化失败：" + e.getMessage());
        }
    }
}
