package com.zehong.system.service.impl;

import java.util.List;

import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.SecurityUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.service.ISysConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.zehong.system.mapper.TTransactionProjectMapper;
import com.zehong.system.domain.TTransactionProject;
import com.zehong.system.service.ITTransactionProjectService;

/**
 * 交易项目Service业务层处理
 * 
 * @author zehong
 * @date 2023-06-06
 */
@Service
public class TTransactionProjectServiceImpl implements ITTransactionProjectService 
{
    private static final Logger log = LoggerFactory.getLogger(TTransactionProjectServiceImpl.class);
    @Autowired
    private TTransactionProjectMapper tTransactionProjectMapper;

    @Autowired
    private ISysConfigService configService;

    /**
     * 查询交易项目
     * 
     * @param transactionProjectId 交易项目ID
     * @return 交易项目
     */
    @Override
    public TTransactionProject selectTTransactionProjectById(Long transactionProjectId)
    {
        return tTransactionProjectMapper.selectTTransactionProjectById(transactionProjectId);
    }

    /**
     * 查询交易项目列表
     * 
     * @param tTransactionProject 交易项目
     * @return 交易项目
     */
    @Override
    public List<TTransactionProject> selectTTransactionProjectList(TTransactionProject tTransactionProject)
    {
        return tTransactionProjectMapper.selectTTransactionProjectList(tTransactionProject);
    }

    /**
     * 新增交易项目
     * 
     * @param tTransactionProject 交易项目
     * @return 结果
     */
    @Override
    public int insertTTransactionProject(TTransactionProject tTransactionProject)
    {
        tTransactionProject.setCreateTime(DateUtils.getNowDate());
        return tTransactionProjectMapper.insertTTransactionProject(tTransactionProject);
    }

    /**
     * 修改交易项目
     * 
     * @param tTransactionProject 交易项目
     * @return 结果
     */
    @Override
    public int updateTTransactionProject(TTransactionProject tTransactionProject)
    {
        tTransactionProject.setUpdateTime(DateUtils.getNowDate());
        return tTransactionProjectMapper.updateTTransactionProject(tTransactionProject);
    }

    /**
     * 批量删除交易项目
     * 
     * @param transactionProjectIds 需要删除的交易项目ID
     * @return 结果
     */
    @Override
    public int deleteTTransactionProjectByIds(Long[] transactionProjectIds)
    {
        return tTransactionProjectMapper.deleteTTransactionProjectByIds(transactionProjectIds);
    }

    /**
     * 删除交易项目信息
     * 
     * @param transactionProjectId 交易项目ID
     * @return 结果
     */
    @Override
    public int deleteTTransactionProjectById(Long transactionProjectId)
    {
        return tTransactionProjectMapper.deleteTTransactionProjectById(transactionProjectId);
    }

    /**
     * 导入数据
     *
     * @param transactionList 数据列表
     * @param isUpdateSupport 是否更新支持，如果已存在，则进行更新数据
     * @return 结果
     */
    @Override
    public String importTransaction(List<TTransactionProject> transactionList, Boolean isUpdateSupport)
    {
        if (StringUtils.isNull(transactionList) || transactionList.size() == 0)
        {
            throw new CustomException("导入数据不能为空！");
        }
        int successNum = 0;
        int failureNum = 0;
        StringBuilder successMsg = new StringBuilder();
        StringBuilder failureMsg = new StringBuilder();
        for (TTransactionProject transaction : transactionList)
        {
            try
            {
                // 验证是否存在这个用户
                TTransactionProject project = tTransactionProjectMapper.selectTransactionProjectByName(transaction.getTransactionProjectName());
                if (StringUtils.isNull(project))
                {
                    this.insertTTransactionProject(transaction);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、交易项目 " + transaction.getTransactionProjectName() + " 导入成功");
                }
                else if (isUpdateSupport)
                {
                    transaction.setTransactionProjectId(project.getTransactionProjectId());
                    this.updateTTransactionProject(transaction);
                    successNum++;
                    successMsg.append("<br/>" + successNum + "、交易项目 " + transaction.getTransactionProjectName() + " 更新成功");
                }
                else
                {
                    failureNum++;
                    failureMsg.append("<br/>" + failureNum + "、交易项目 " + transaction.getTransactionProjectName() + " 已存在");
                }
            }
            catch (Exception e)
            {
                failureNum++;
                String msg = "<br/>" + failureNum + "、交易项目 " + transaction.getTransactionProjectName() + " 导入失败：";
                failureMsg.append(msg + e.getMessage());
                log.error(msg, e);
            }
        }
        if (failureNum > 0)
        {
            failureMsg.insert(0, "很抱歉，导入失败！共 " + failureNum + " 条数据格式不正确，错误如下：");
            throw new CustomException(failureMsg.toString());
        }
        else
        {
            successMsg.insert(0, "恭喜您，数据已全部导入成功！共 " + successNum + " 条，数据如下：");
        }
        return successMsg.toString();
    }
}
