package com.zehong.system.service.impl;

import com.zehong.common.core.domain.entity.SysRole;
import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.SecurityUtils;
import com.zehong.system.domain.TTradeProject;
import com.zehong.system.mapper.TTradeProjectMapper;
import com.zehong.system.service.ITTradeProjectService;
import com.zehong.system.service.impl.tradeRoles.TradeRoles;
import com.zehong.system.service.impl.tradeRoles.roles.*;
import com.zehong.system.service.impl.tradeSettlement.SettlementTrade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 交易项目Service业务层处理
 * 
 * @author zehong
 * @date 2023-06-08
 */
@Service
public class TTradeProjectServiceImpl implements ITTradeProjectService 
{
    @Autowired
    private TTradeProjectMapper tTradeProjectMapper;

    @Resource
    private SettlementTrade settlementTrade;

    /**
     * 查询交易项目
     * 
     * @param tradeId 交易项目ID
     * @return 交易项目
     */
    @Override
    public TTradeProject selectTTradeProjectById(Long tradeId)
    {
        return tTradeProjectMapper.selectTTradeProjectById(tradeId);
    }

    /**
     * 查询交易项目列表
     * 
     * @param tTradeProject 交易项目
     * @return 交易项目
     */
    @Override
    public List<TTradeProject> selectTTradeProjectList(TTradeProject tTradeProject)
    {
        return tTradeProjectMapper.selectTTradeProjectList(tTradeProject);
    }

    /**
     * 根据角色查询数据
     * @param tTradeProject 交易实体
     * @return
     */
    @Override
    public List<TTradeProject> selectTTradeProjectListByRoles(TTradeProject tTradeProject){
        judgeRole(tTradeProject).setQueryParams();
        List<TTradeProject> tradeProjectList = tTradeProjectMapper.selectTTradeProjectList(tTradeProject);
        for(TTradeProject project :tradeProjectList){
            judgeRole(project).setOperators();
        }
        return tradeProjectList;
    }

    /**
     * 判断角色
     * @param tTradeProject
     * @return TradeRoles
     */
    private TradeRoles judgeRole(TTradeProject tTradeProject) {
        List<SysRole> roles = SecurityUtils.getLoginUser().getUser().getRoles();
        List<String> roleKeys = roles.stream().map(item -> item.getRoleKey()).collect(Collectors.toList());

        if(roleKeys.contains("admin")){
            return new TradeAdmin(tTradeProject);
        }
        //双权限
        if(roleKeys.contains("transactor") && roleKeys.contains("deptLeader")){
            return new DoubleRoles(tTradeProject);
        }
        //经办人
        if(roleKeys.contains("transactor")){
            return new Transactor(tTradeProject);
        }
        //部长
        if(roleKeys.contains("deptLeader")){
            return new DeptLeader(tTradeProject);
        }
        return new OtherRoles(tTradeProject);
    }

    /**
     * 新增交易项目
     * 
     * @param tTradeProject 交易项目
     * @return 结果
     */
    @Override
    public int insertTTradeProject(TTradeProject tTradeProject)
    {
        tTradeProject.setReceiptNum("ZEHONG" + DateUtils.dateTimeNow());
        SysUser user = SecurityUtils.getLoginUser().getUser();
        tTradeProject.setApplyId(user.getUserId());
        tTradeProject.setApplyDeptId(user.getDeptId());
        tTradeProject.setCreateTime(DateUtils.getNowDate());
        return tTradeProjectMapper.insertTTradeProject(tTradeProject);
    }


    /**
     * 修改交易项目
     * 
     * @param tTradeProject 交易项目
     * @return 结果
     */
    @Override
    public int updateTTradeProject(TTradeProject tTradeProject)
    {
        tTradeProject.setUpdateTime(DateUtils.getNowDate());
        return tTradeProjectMapper.updateTTradeProject(tTradeProject);
    }
    /**
     * 批量删除交易项目
     * 
     * @param tradeIds 需要删除的交易项目ID
     * @return 结果
     */
    @Override
    public int deleteTTradeProjectByIds(Long[] tradeIds)
    {
        return tTradeProjectMapper.deleteTTradeProjectByIds(tradeIds);
    }

    /**
     * 删除交易项目信息
     * 
     * @param tradeId 交易项目ID
     * @return 结果
     */
    @Override
    public int deleteTTradeProjectById(Long tradeId)
    {
        return tTradeProjectMapper.deleteTTradeProjectById(tradeId);
    }

    /**
     * 交易结算
     * @param tTradeProject
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public int settlementTrade(TTradeProject tTradeProject){
        TTradeProject tradeInfo = tTradeProjectMapper.selectTTradeProjectById(tTradeProject.getTradeId());
        //交易结算
        settlementTrade.settleAbleAmount(tradeInfo);
        tradeInfo.setTradeStatus(tTradeProject.getTradeStatus());
        tradeInfo.setApplyDeptManagerId(tTradeProject.getApplyDeptManagerId());
        tradeInfo.setSettlementTime(new Date());
        return tTradeProjectMapper.updateTTradeProject(tradeInfo);
    }



    @Override
    public List<TTradeProject> getIncomeOrPayableList(TTradeProject tTradeProject) {
        return tTradeProjectMapper.getIncomeOrPayableList(tTradeProject);
    }
}
