package com.zehong.system.service.impl;

import com.zehong.common.core.domain.entity.SysRole;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.SecurityUtils;
import com.zehong.system.domain.TDebitCredit;
import com.zehong.system.mapper.TDebitCreditMapper;
import com.zehong.system.service.ITDebitCreditService;
import com.zehong.system.service.impl.debitCredit.DebitCreditRoles;
import com.zehong.system.service.impl.debitCredit.roles.*;
import com.zehong.system.service.impl.debitSettlement.SettlementDebit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 借贷Service业务层处理
 * 
 * @author zehong
 * @date 2023-06-20
 */
@Service
public class TDebitCreditServiceImpl implements ITDebitCreditService 
{
    @Autowired
    private TDebitCreditMapper tDebitCreditMapper;

    @Autowired
    private SettlementDebit settlementDebit;

    /**
     * 查询借贷
     * 
     * @param debitId 借贷ID
     * @return 借贷
     */
    @Override
    public TDebitCredit selectTDebitCreditById(Long debitId)
    {
        return tDebitCreditMapper.selectTDebitCreditById(debitId);
    }

    /**
     * 查询借贷列表
     * 
     * @param tDebitCredit 借贷
     * @return 借贷
     */
    @Override
    public List<TDebitCredit> selectTDebitCreditList(TDebitCredit tDebitCredit)
    {
        return tDebitCreditMapper.selectTDebitCreditList(tDebitCredit);
    }

    /**
     * 根据角色查询借贷列表
     * @param tDebitCredit 借贷
     * @return 借贷集合
     */
    @Override
    public List<TDebitCredit> selectDebitListByRoles(TDebitCredit tDebitCredit){
        judgeRole(tDebitCredit).setParams();
        List<TDebitCredit> tDebitCreditList = tDebitCreditMapper.selectTDebitCreditList(tDebitCredit);
        tDebitCreditList.stream().forEach(item ->{
            judgeRole(item).setOperators();
        });
        return tDebitCreditList;
    }

    /**
     * 判断角色
     * @param tDebitCredit 借贷实体
     * @return DebitCreditRoles
     */
    private DebitCreditRoles judgeRole(TDebitCredit tDebitCredit) {
        List<SysRole> roles = SecurityUtils.getLoginUser().getUser().getRoles();
        List<String> roleKeys = roles.stream().map(item -> item.getRoleKey()).collect(Collectors.toList());

        if(roleKeys.contains("admin")){
            return new DebitCreditAdmin(tDebitCredit);
        }
        //双权限
        if(roleKeys.contains("calculator") && roleKeys.contains("deptLeader")){
            return new DoubleRoles(tDebitCredit);
        }
        //部长
        if(roleKeys.contains("deptLeader")){
            return new DeptLeader(tDebitCredit);
        }
        //核算部
        if(roleKeys.contains("calculator")){
            return new Calculator(tDebitCredit);
        }
        return new OtherRoles(tDebitCredit);
    }

    /**
     * 新增借贷
     * 
     * @param tDebitCredit 借贷
     * @return 结果
     */
    @Override
    public int insertTDebitCredit(TDebitCredit tDebitCredit)
    {
        tDebitCredit.setCreateTime(DateUtils.getNowDate());
        return tDebitCreditMapper.insertTDebitCredit(tDebitCredit);
    }

    /**
     * 修改借贷
     * 
     * @param tDebitCredit 借贷
     * @return 结果
     */
    @Override
    public int updateTDebitCredit(TDebitCredit tDebitCredit)
    {
        tDebitCredit.setUpdateTime(DateUtils.getNowDate());
        return tDebitCreditMapper.updateTDebitCredit(tDebitCredit);
    }

    /**
     * 批量删除借贷
     * 
     * @param debitIds 需要删除的借贷ID
     * @return 结果
     */
    @Override
    public int deleteTDebitCreditByIds(Long[] debitIds)
    {
        return tDebitCreditMapper.deleteTDebitCreditByIds(debitIds);
    }

    /**
     * 删除借贷信息
     * 
     * @param debitId 借贷ID
     * @return 结果
     */
    @Override
    public int deleteTDebitCreditById(Long debitId)
    {
        return tDebitCreditMapper.deleteTDebitCreditById(debitId);
    }

    /**
     * 借贷结算
     * @param tDebitCredit 借贷实体
     */
    @Transactional(rollbackFor = Exception.class)
    public int settlementDebitCredit(TDebitCredit tDebitCredit){
        //结算
        settlementDebit.settlementDebit(tDebitCredit);
        tDebitCredit.setUpdateTime(DateUtils.getNowDate());
        return tDebitCreditMapper.updateTDebitCredit(tDebitCredit);
    }

    /**
     * 还款
     * @param tDebitCredit 借贷实体
     */
    @Transactional(rollbackFor = Exception.class)
    public int repayDebitCredit(TDebitCredit tDebitCredit){
        //结算
        settlementDebit.repayment(tDebitCredit);
        tDebitCredit.setUpdateTime(DateUtils.getNowDate());
        return tDebitCreditMapper.updateTDebitCredit(tDebitCredit);
    }
}
