package com.zehong.web.controller.transaction;

import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.core.domain.model.LoginUser;
import com.zehong.common.core.page.TableDataInfo;
import com.zehong.common.enums.BusinessType;
import com.zehong.common.utils.ServletUtils;
import com.zehong.common.utils.StringUtils;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.framework.web.service.TokenService;
import com.zehong.system.domain.TTransactionProject;
import com.zehong.system.service.ITTransactionProjectService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;

/**
 * 交易项目Controller
 * 
 * @author zehong
 * @date 2023-06-05
 */
@RestController
@RequestMapping("/transaction/project")
public class TTransactionProjectController extends BaseController
{
    @Autowired
    private ITTransactionProjectService tTransactionProjectService;
    @Autowired
    private TokenService tokenService;

    /**
     * 查询交易项目列表
     */
    @PreAuthorize("@ss.hasPermi('system:project:list')")
    @GetMapping("/list")
    public TableDataInfo list(TTransactionProject tTransactionProject)
    {
        startPage();
        // 获取当前用户
        SysUser user = tokenService.getLoginUser(ServletUtils.getRequest()).getUser();
        if (StringUtils.isNotNull(user) && !user.isAdmin()) {
            tTransactionProject.setDeptId(user.getDeptId());
        }
        List<TTransactionProject> list = tTransactionProjectService.selectTTransactionProjectList(tTransactionProject);
        return getDataTable(list);

    }


    /**
     * 获取所有交易项目
     * @param tTransactionProject
     * @return
     */
    @GetMapping("/getTransactionProjectList")
    public AjaxResult getTransactionProjectList(TTransactionProject tTransactionProject){
        List<TTransactionProject> list = tTransactionProjectService.selectTTransactionProjectList(tTransactionProject);
        return AjaxResult.success(list);
    }

    /**
     * 导出交易项目列表
     */
    @PreAuthorize("@ss.hasPermi('system:project:export')")
    @Log(title = "交易项目", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TTransactionProject tTransactionProject)
    {
        // 获取当前用户
        SysUser user = tokenService.getLoginUser(ServletUtils.getRequest()).getUser();
        if (StringUtils.isNotNull(user) && !user.isAdmin()) {
            tTransactionProject.setDeptId(user.getDeptId());
        }
        List<TTransactionProject> list = tTransactionProjectService.selectTTransactionProjectList(tTransactionProject);
        ExcelUtil<TTransactionProject> util = new ExcelUtil<TTransactionProject>(TTransactionProject.class);
        return util.exportExcel(list, "交易项目数据");
    }

    @Log(title = "交易项目维护", businessType = BusinessType.IMPORT)
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception
    {
        ExcelUtil<TTransactionProject> util = new ExcelUtil<TTransactionProject>(TTransactionProject.class);
        List<TTransactionProject> transactionList = util.importExcel(file.getInputStream());
        String message = tTransactionProjectService.importTransaction(transactionList, updateSupport);
        return AjaxResult.success(message);
    }

    @GetMapping("/importTemplate")
    public AjaxResult importTemplate()
    {
        ExcelUtil<TTransactionProject> util = new ExcelUtil<TTransactionProject>(TTransactionProject.class);
        return util.importTemplateExcel("交易项目数据");
    }

    /**
     * 获取交易项目详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:project:query')")
    @GetMapping(value = "/{transactionProjectId}")
    public AjaxResult getInfo(@PathVariable("transactionProjectId") Long transactionProjectId)
    {
        return AjaxResult.success(tTransactionProjectService.selectTTransactionProjectById(transactionProjectId));
    }

    /**
     * 新增交易项目
     */
    @PreAuthorize("@ss.hasPermi('system:project:add')")
    @Log(title = "交易项目", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TTransactionProject tTransactionProject)
    {
        // 获取当前用户
        SysUser user = tokenService.getLoginUser(ServletUtils.getRequest()).getUser();
        if (StringUtils.isNotNull(user) && !user.isAdmin()) {
            tTransactionProject.setDeptId(user.getDeptId());
        }
        return toAjax(tTransactionProjectService.insertTTransactionProject(tTransactionProject));
    }

    /**
     * 修改交易项目
     */
    @PreAuthorize("@ss.hasPermi('system:project:edit')")
    @Log(title = "交易项目", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TTransactionProject tTransactionProject)
    {
        return toAjax(tTransactionProjectService.updateTTransactionProject(tTransactionProject));
    }

    /**
     * 删除交易项目
     */
    @PreAuthorize("@ss.hasPermi('system:project:remove')")
    @Log(title = "交易项目", businessType = BusinessType.DELETE)
	@DeleteMapping("/{transactionProjectIds}")
    public AjaxResult remove(@PathVariable Long[] transactionProjectIds)
    {
        return toAjax(tTransactionProjectService.deleteTTransactionProjectByIds(transactionProjectIds));
    }
}
