package com.zehong.web.controller.transaction;

import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.core.page.TableDataInfo;
import com.zehong.common.enums.BusinessType;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.system.domain.TTradeProject;
import com.zehong.system.service.ITTradeProjectService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 交易项目Controller
 *
 * @author zehong
 * @date 2023-06-08
 */
@RestController
@RequestMapping("/trade/project")
public class TTradeProjectController extends BaseController
{
    @Autowired
    private ITTradeProjectService tTradeProjectService;

    /**
     * 查询交易项目列表
     */
    @PreAuthorize("@ss.hasPermi('trade:project:list')")
    @GetMapping("/list")
    public TableDataInfo list(TTradeProject tTradeProject)
    {
        startPage();
        List<TTradeProject> list = tTradeProjectService.selectTTradeProjectList(tTradeProject);
        return getDataTable(list);
    }

    @GetMapping("/selectTTradeProjectListByRoles")
    public TableDataInfo selectTTradeProjectListByRoles(TTradeProject tTradeProject){
        startPage();
        List<TTradeProject> list = tTradeProjectService.selectTTradeProjectListByRoles(tTradeProject);
        return getDataTable(list);
    }

    /**
     * 导出交易项目列表
     */
    @PreAuthorize("@ss.hasPermi('trade:project:export')")
    @Log(title = "交易项目", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TTradeProject tTradeProject)
    {
        List<TTradeProject> list = tTradeProjectService.selectTTradeProjectList(tTradeProject);
        ExcelUtil<TTradeProject> util = new ExcelUtil<TTradeProject>(TTradeProject.class);
        return util.exportExcel(list, "交易项目数据");
    }

    /**
     * 获取交易项目详细信息
     */
    @PreAuthorize("@ss.hasPermi('trade:project:query')")
    @GetMapping(value = "/{tradeId}")
    public AjaxResult getInfo(@PathVariable("tradeId") Long tradeId)
    {
        return AjaxResult.success(tTradeProjectService.selectTTradeProjectById(tradeId));
    }

    /**
     * 新增交易项目
     */
    @PreAuthorize("@ss.hasPermi('trade:project:add')")
    @Log(title = "交易项目", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TTradeProject tTradeProject)
    {
        return toAjax(tTradeProjectService.insertTTradeProject(tTradeProject));
    }

    /**
     * 修改交易项目
     */
    @PreAuthorize("@ss.hasPermi('trade:project:edit')")
    @Log(title = "交易项目", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TTradeProject tTradeProject)
    {
        return toAjax(tTradeProjectService.updateTTradeProject(tTradeProject));
    }

    /**
     * 删除交易项目
     */
    @PreAuthorize("@ss.hasPermi('trade:project:remove')")
    @Log(title = "交易项目", businessType = BusinessType.DELETE)
	@DeleteMapping("/{tradeIds}")
    public AjaxResult remove(@PathVariable Long[] tradeIds)
    {
        return toAjax(tTradeProjectService.deleteTTradeProjectByIds(tradeIds));
    }

    /**
     * 交易结算
     * @param tTradeProject
     * @return
     */
    @PostMapping("/settlementTrade")
    public AjaxResult settlementTrade(@RequestBody TTradeProject tTradeProject){
        return toAjax(tTradeProjectService.settlementTrade(tTradeProject));
    }

    @GetMapping("/getIncomeOrPayableList")
    public TableDataInfo getIncomeOrPayableList(TTradeProject tTradeProject)
    {
        startPage();
        List<TTradeProject> list = tTradeProjectService.getIncomeOrPayableList(tTradeProject);
        return getDataTable(list);
    }
}
