package com.zehong.web.controller.debit;

import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.core.exception.BusinessException;
import com.zehong.common.core.page.TableDataInfo;
import com.zehong.common.enums.BusinessType;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.system.domain.TDebitCredit;
import com.zehong.system.service.ITAccountService;
import com.zehong.system.service.ITDebitCreditService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 借贷Controller
 * 
 * @author zehong
 * @date 2023-06-20
 */
@RestController
@RequestMapping("/debit/credit")
public class TDebitCreditController extends BaseController
{
    @Autowired
    private ITDebitCreditService tDebitCreditService;

    @Autowired
    private ITAccountService itAccountService;

    /**
     * 查询借贷列表
     */
    @PreAuthorize("@ss.hasPermi('system:credit:list')")
    @GetMapping("/list")
    public TableDataInfo list(TDebitCredit tDebitCredit)
    {
        startPage();
        List<TDebitCredit> list = tDebitCreditService.selectTDebitCreditList(tDebitCredit);
        return getDataTable(list);
    }

    @PostMapping("/selectDebitListByRoles")
    public TableDataInfo selectDebitListByRoles(@RequestBody TDebitCredit tDebitCredit){
        startPage();
        List<TDebitCredit> list = tDebitCreditService.selectDebitListByRoles(tDebitCredit);
        return getDataTable(list);
    }

    /**
     * 导出借贷列表
     */
    @PreAuthorize("@ss.hasPermi('system:credit:export')")
    @Log(title = "借贷", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TDebitCredit tDebitCredit)
    {
        List<TDebitCredit> list = tDebitCreditService.selectTDebitCreditList(tDebitCredit);
        ExcelUtil<TDebitCredit> util = new ExcelUtil<TDebitCredit>(TDebitCredit.class);
        return util.exportExcel(list, "借贷数据");
    }

    /**
     * 获取借贷详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:credit:query')")
    @GetMapping(value = "/{debitId}")
    public AjaxResult getInfo(@PathVariable("debitId") Long debitId)
    {
        return AjaxResult.success(tDebitCreditService.selectTDebitCreditById(debitId));
    }

    /**
     * 新增借贷
     */
    @PreAuthorize("@ss.hasPermi('system:credit:add')")
    @Log(title = "借贷", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TDebitCredit tDebitCredit)
    {
        if(!itAccountService.checkAccountCanTrade(tDebitCredit.getLendDeptId(),tDebitCredit.getLittleTotal())){
            return AjaxResult.error("出借部门账户可用金额不足！");
        }
        return toAjax(tDebitCreditService.insertTDebitCredit(tDebitCredit));
    }

    /**
     * 修改借贷
     */
    @PreAuthorize("@ss.hasPermi('system:credit:edit')")
    @Log(title = "借贷", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TDebitCredit tDebitCredit)
    {
        return toAjax(tDebitCreditService.updateTDebitCredit(tDebitCredit));
    }

    /**
     * 删除借贷
     */
    @PreAuthorize("@ss.hasPermi('system:credit:remove')")
    @Log(title = "借贷", businessType = BusinessType.DELETE)
	@DeleteMapping("/{debitIds}")
    public AjaxResult remove(@PathVariable Long[] debitIds)
    {
        return toAjax(tDebitCreditService.deleteTDebitCreditByIds(debitIds));
    }

    /**
     * 借贷结算
     * @param tDebitCredit 借贷实体
     */
    @PostMapping("/settlementDebitCredit")
    public AjaxResult settlementDebitCredit(@RequestBody TDebitCredit tDebitCredit){
        try{
            return toAjax(tDebitCreditService.settlementDebitCredit(tDebitCredit));
        }catch (BusinessException e){
            return AjaxResult.error(e.getMessage());
        }
    }

    /**
     * 还款
     * @param tDebitCredit 借贷实体
     */
    @PostMapping("/repayDebitCredit")
    public AjaxResult repayDebitCredit(@RequestBody TDebitCredit tDebitCredit){
        try{
            return toAjax(tDebitCreditService.repayDebitCredit(tDebitCredit));
        }catch (BusinessException e){
            return AjaxResult.error(e.getMessage());
        }
    }
}
