package com.zehong.system.service.impl;

import com.zehong.common.exception.CustomException;
import com.zehong.common.utils.DateUtils;
import com.zehong.system.domain.TPipePoint;
import com.zehong.system.domain.TPipeTask;
import com.zehong.system.domain.vo.TaskInspectStatisticsByPipVo;
import com.zehong.system.mapper.TPipePointMapper;
import com.zehong.system.mapper.TPipeTaskInspectMapper;
import com.zehong.system.mapper.TPipeTaskMapper;
import com.zehong.system.service.ITPipeTaskService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 管线巡线任务Service业务层处理
 * 
 * @author zehong
 * @date 2023-11-17
 */
@Service
public class TPipeTaskServiceImpl implements ITPipeTaskService 
{
    @Autowired
    private TPipeTaskMapper tPipeTaskMapper;
    @Autowired
    private TPipeTaskInspectMapper tPipeTaskInspectMapper;
    @Autowired
    private TPipePointMapper tPipePointMapper;

    /**
     * 查询管线巡线任务
     * 
     * @param id 管线巡线任务ID
     * @return 管线巡线任务
     */
    @Override
    public TPipeTask selectTPipeTaskById(Long id)
    {
        return tPipeTaskMapper.selectTPipeTaskById(id);
    }

    /**
     * 查询管线巡线任务列表
     * 
     * @param tPipeTask 管线巡线任务
     * @return 管线巡线任务
     */
    @Override
    public List<TPipeTask> selectTPipeTaskList(TPipeTask tPipeTask)
    {
        return tPipeTaskMapper.selectTPipeTaskList(tPipeTask);
    }
    @Override
    public List<TPipeTask> selectTwxPipeTaskList(TPipeTask tPipeTask)
    {
        return tPipeTaskMapper.selectTwxPipeTaskList(tPipeTask);
    }

    /**
     * 新增管线巡线任务
     * 
     * @param tPipeTask 管线巡线任务
     * @return 结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int insertTPipeTask(TPipeTask tPipeTask){
        if("0".equals(tPipeTask.getPipeTaskType())){
            return addHeightTask(tPipeTask);
        }
        return addCountryTask(tPipeTask);
    }

    /**
     * 新增中高压管线任务
     * @param tPipeTask 任务实体
     * @return
     */
    private int addHeightTask(TPipeTask tPipeTask){
        LocalDate startDate = LocalDate.parse(DateUtils.dateTime(tPipeTask.getPlanExecuteDate().get(0)));
        LocalDate endDate = LocalDate.parse(DateUtils.dateTime(tPipeTask.getPlanExecuteDate().get(1)));
        long numOfDays = ChronoUnit.DAYS.between(startDate, endDate) + 1;
        int num = 0;
        for (int i = 0; i < numOfDays; i++) {
            LocalDate currentDate = startDate.plusDays(i);
            //String format = currentDate.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            tPipeTask.setCreateTime(DateUtils.getNowDate());
            tPipeTask.setPushTime(Date.from(currentDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant()));
            TPipeTask query = new TPipeTask();
            query.setPushTime(Date.from(currentDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant()));
            query.setPipeTaskType(tPipeTask.getPipeTaskType());
            List<TPipeTask> taskList = tPipeTaskMapper.selectTPipeTaskList(query);
            if(CollectionUtils.isEmpty(taskList)){
                tPipeTaskMapper.insertTPipeTask(tPipeTask);
                List<TPipePoint> pointList = tPipePointMapper.selectAllpoint(tPipeTask.getPipeTaskType());
                if(CollectionUtils.isEmpty(pointList)) throw new CustomException("未设置管线、打卡点或管线未分配人员");
                tPipeTaskInspectMapper.inserts(pointList,tPipeTask.getId(),tPipeTask.getCreateTime());
                num++;
            }
        }
        return num;
    }

    /**
     * 新增村内管线任务
     * @param tPipeTask 任务实体
     * @return
     */
    private int addCountryTask(TPipeTask tPipeTask){
        tPipeTask.setPushTime(tPipeTask.getPlanExecuteDate().get(0));
        tPipeTask.setTaskEndTime(tPipeTask.getPlanExecuteDate().get(1));
        tPipeTask.setCreateTime(DateUtils.getNowDate());
        int i = tPipeTaskMapper.insertTPipeTask(tPipeTask);
        List<TPipePoint> pointList = tPipePointMapper.selectAllpoint(tPipeTask.getPipeTaskType());
        if(CollectionUtils.isEmpty(pointList)) throw new CustomException("未设置管线、打卡点或管线未分配人员");
        tPipeTaskInspectMapper.inserts(pointList,tPipeTask.getId(),tPipeTask.getCreateTime());
        return i;
    }

    /**
     * 修改管线巡线任务
     * 
     * @param tPipeTask 管线巡线任务
     * @return 结果
     */
    @Override
    public int updateTPipeTask(TPipeTask tPipeTask)
    {
        return tPipeTaskMapper.updateTPipeTask(tPipeTask);
    }

    /**
     * 批量删除管线巡线任务
     * 
     * @param ids 需要删除的管线巡线任务ID
     * @return 结果
     */
    @Override
    public int deleteTPipeTaskByIds(Long[] ids)
    {
        return tPipeTaskMapper.deleteTPipeTaskByIds(ids);
    }

    /**
     * 删除管线巡线任务信息
     * 
     * @param id 管线巡线任务ID
     * @return 结果
     */
    @Override
    public int deleteTPipeTaskById(Long id)
    {
        return tPipeTaskMapper.deleteTPipeTaskById(id);
    }


    /**
     * 巡线人巡线点统计
     * @param
     * @return
     */
    @Override
    public List<Map<String,Object>> selectTaskInspectStatisticsByUser(Map<String,Object> param){
        return tPipeTaskMapper.selectTaskInspectStatisticsByUser(param);
    }

    /**
     * 管道巡检点统计
     * @param param 入参
     * @return
     */
    @Override
    public List<Map<String,Object>> selectTaskInspectStatisticsByPipe(TaskInspectStatisticsByPipVo param){
        return tPipeTaskMapper.selectTaskInspectStatisticsByPipe(param);
    }
}
