package com.zehong.system.service.impl;

import java.util.Date;
import java.util.List;

import com.alibaba.fastjson.JSONObject;
import com.zehong.common.core.redis.RedisCache;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.zehong.system.mapper.TGasDataMapper;
import com.zehong.system.domain.TGasData;
import com.zehong.system.service.ITGasDataService;

/**
 * 监测设备数据Service业务层处理
 * 
 * @author zehong
 * @date 2024-03-02
 */
@Service
public class TGasDataServiceImpl implements ITGasDataService 
{
    @Autowired
    private TGasDataMapper tGasDataMapper;

    @Autowired
    private RedisCache redisCache;

    /**
     * 查询监测设备数据
     * 
     * @param id 监测设备数据ID
     * @return 监测设备数据
     */
    @Override
    public TGasData selectTGasDataById(Long id)
    {
        return tGasDataMapper.selectTGasDataById(id);
    }

    /**
     * 查询监测设备数据列表
     * 
     * @param tGasData 监测设备数据
     * @return 监测设备数据
     */
    @Override
    public List<TGasData> selectTGasDataList(TGasData tGasData){
        List<TGasData> gasDataList = tGasDataMapper.selectTGasDataList(tGasData);
        gasDataList.stream().forEach(item ->{
            JSONObject gasData = redisCache.getCacheMapValue("devices_report",item.getNumber());
            if(null != gasData){
                item.setReportStatus(gasData.getString("status"));
                item.setReportParam(gasData.getString("value") + item.getUnit());
                item.setReportTime(gasData.getString("time"));
            }
        });
        return gasDataList;
    }

    /**
     * 获取上报状态
     *1 => '在线',
     *2 => '自检',
     *3 => '传感器故障',
     *4 => '报警',
     *5 => '低报',
     *6 => '高报',
     *7 => '通信故障',
     *8 => '超量程',
     *9 => '离线',
     *10 => '电量低',
     *11 => '主电故障',
     *12 => '备电故障',
     *13 => '无此节点',
     *14 => '低电压',
     *15 => '故障',
     *16 => '报警联动',
     *17 => '低低报',
     *18 => '高高报',
     *19 => '水位淹没报警',
     * @param statusCode 状态码
     * @return
     */
    private String getReportStatus(String statusCode){
        switch (statusCode){
            case "0":
            case "2":
                return "连接";
            case "1":
                return "在线";
            case "3":
            case "7":
            case "8":
            case "10":
            case "11":
            case "12":
            case "13":
            case "14":
            case "15":
                return "异常";
            case "4":
            case "5":
            case "6":
            case "16":
            case "17":
            case "18":
            case "19":
                return "报警";
            case "9":
                return "离线";
            default:
                return "";

        }
    }

    /**
     * 新增监测设备数据
     * 
     * @param tGasData 监测设备数据
     * @return 结果
     */
    @Override
    public int insertTGasData(TGasData tGasData) {
        tGasData.setCreatedAt(new Date());
        return tGasDataMapper.insertTGasData(tGasData);
    }

    /**
     * 修改监测设备数据
     * 
     * @param tGasData 监测设备数据
     * @return 结果
     */
    @Override
    public int updateTGasData(TGasData tGasData)
    {
        return tGasDataMapper.updateTGasData(tGasData);
    }

    /**
     * 批量删除监测设备数据
     * 
     * @param ids 需要删除的监测设备数据ID
     * @return 结果
     */
    @Override
    public int deleteTGasDataByIds(Long[] ids)
    {
        return tGasDataMapper.deleteTGasDataByIds(ids);
    }

    /**
     * 删除监测设备数据信息
     * 
     * @param id 监测设备数据ID
     * @return 结果
     */
    @Override
    public int deleteTGasDataById(Long id)
    {
        return tGasDataMapper.deleteTGasDataById(id);
    }
}
