package com.zehong.system.service.impl;

import com.github.pagehelper.PageInfo;
import com.zehong.common.core.domain.entity.SysDictData;
import com.zehong.common.core.domain.entity.SysUser;
import com.zehong.common.utils.DateUtils;
import com.zehong.common.utils.PageInfoUtil;
import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.*;
import com.zehong.system.domain.form.HiddenTroubleForm;
import com.zehong.system.domain.vo.HiddenTroubleVo;
import com.zehong.system.mapper.*;
import com.zehong.system.service.ISysDictTypeService;
import com.zehong.system.service.ITHiddenTroubleService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * 隐患信息Service业务层处理
 * 
 * @author zehong
 * @date 2021-07-21
 */
@Service
public class THiddenTroubleServiceImpl implements ITHiddenTroubleService 
{
    @Autowired
    private THiddenTroubleMapper tHiddenTroubleMapper;
    @Autowired
    private TDeviceInfoMapper tDeviceInfoMapper;
    @Autowired
    private TPipeMapper tPipeMapper;
    @Autowired
    private SysUserMapper sysUserMapper;
    @Autowired
    private TWorkOrderMapper tWorkOrderMapper;
    @Autowired
    private TOrderFeedbackMapper tOrderFeedbackMapper;
    @Autowired
    private ISysDictTypeService iSysDictTypeService;

    /**
     * 查询隐患信息
     * 
     * @param troubleId 隐患信息ID
     * @return 隐患信息
     */
    @Override
    public HiddenTroubleVo selectTHiddenTroubleById(int troubleId) throws Exception
    {
        HiddenTroubleVo hiddenTroubleVo = new HiddenTroubleVo();
        THiddenTrouble tHiddenTrouble = tHiddenTroubleMapper.selectTHiddenTroubleById(troubleId);
        BeanUtils.copyProperties(tHiddenTrouble, hiddenTroubleVo);

        SysUser sysUser = sysUserMapper.selectUserById(tHiddenTrouble.getReportMan());
        hiddenTroubleVo.setReportManName(sysUser.getNickName());

        if(tHiddenTrouble.getDeviceId() != null){
            if("0".equals(tHiddenTrouble.getDeviceType())){
                TPipe pipe = tPipeMapper.selectTPipeById(tHiddenTrouble.getDeviceId());
                hiddenTroubleVo.setDeviceName(pipe.getPipeName());
            } else {
                TDeviceInfo device = tDeviceInfoMapper.selectTDeviceInfoById(tHiddenTrouble.getDeviceId());
                hiddenTroubleVo.setDeviceName(device.getDeviceName());
            }
        }

        if(tHiddenTrouble.getOrderId() != null && !"".equals(tHiddenTrouble.getOrderId())) {
            TWorkOrder tWorkOrder = tWorkOrderMapper.selectTWorkOrderById(tHiddenTrouble.getOrderId());
            hiddenTroubleVo.setAllotTime(tWorkOrder.getAllotTime());
            // 获取巡检员姓名
            SysUser appointInspector = sysUserMapper.selectUserById(tWorkOrder.getAppointInspector());
            hiddenTroubleVo.setAppointInspectorName(appointInspector.getNickName());

            List<TOrderFeedback> orderFeedbackList = tOrderFeedbackMapper.selectTOrderFeedbackByOrderId(tHiddenTrouble.getOrderId());
            if(orderFeedbackList.size() != 0){
                hiddenTroubleVo.setFeedbackList(orderFeedbackList);
            }
        }

        return hiddenTroubleVo;
    }

    /**
     * 查询隐患信息列表
     * 
     * @param hiddenTroubleForm 隐患信息
     * @return 隐患信息
     */
    @Override
    public List<HiddenTroubleVo> selectTHiddenTroubleList(HiddenTroubleForm hiddenTroubleForm)
    {
        List<HiddenTroubleVo> list = new ArrayList<>();
        List<THiddenTrouble> tHiddenTroubleList = tHiddenTroubleMapper.selectTHiddenTroubleList(hiddenTroubleForm);

        if(tHiddenTroubleList.size() != 0) {
            for (THiddenTrouble trouble : tHiddenTroubleList) {
                HiddenTroubleVo hiddenTroubleVo = new HiddenTroubleVo();
                BeanUtils.copyProperties(trouble, hiddenTroubleVo);

                SysUser sysUser = sysUserMapper.selectUserById(trouble.getReportMan());
                if(sysUser != null) {
                    hiddenTroubleVo.setReportManName(sysUser.getNickName());
                }

                list.add(hiddenTroubleVo);
            }
        }
        return list;
    }

    /**
     * 查询隐患信息列表
     *
     * @param hiddenTroubleForm 隐患信息
     * @return 隐患信息
     */
    @Override
    public PageInfo<HiddenTroubleVo> selectTHiddenTroublePage(HiddenTroubleForm hiddenTroubleForm)
    {
        List<THiddenTrouble> tHiddenTroubleList = tHiddenTroubleMapper.selectTHiddenTroubleList(hiddenTroubleForm);
        PageInfo<HiddenTroubleVo> pageVo = PageInfoUtil.pageInfo2PageInfoDTO(new PageInfo<>(tHiddenTroubleList),HiddenTroubleVo.class);

        if(pageVo.getList().size() != 0) {
            for (HiddenTroubleVo hiddenTroubleVo : pageVo.getList()) {

                SysUser sysUser = sysUserMapper.selectUserById(hiddenTroubleVo.getReportMan());
                if(sysUser != null) {
                    hiddenTroubleVo.setReportManName(sysUser.getNickName());
                }
            }
        }
        return pageVo;
    }

    /**
     * 地图显示隐患信息列表
     *
     * @param
     * @return 隐患信息集合
     */
    @Override
    public List<HiddenTroubleVo> getMapTHiddenTroubleList(String troubleName)
    {
        List<HiddenTroubleVo> list = new ArrayList<>();
        List<THiddenTrouble> troubleList = tHiddenTroubleMapper.getMapTHiddenTroubleList(troubleName);

        if(troubleList.size() != 0) {
            for (THiddenTrouble temp : troubleList) {
                HiddenTroubleVo hiddenTroubleVo = new HiddenTroubleVo();
                BeanUtils.copyProperties(temp, hiddenTroubleVo);

                SysUser sysUser = sysUserMapper.selectUserById(temp.getReportMan());
                if(sysUser != null) {
                    hiddenTroubleVo.setReportManName(sysUser.getNickName());
                }

                if (StringUtils.isNotEmpty(temp.getTroubleType())) {
                    List<SysDictData> sysDictDataList = iSysDictTypeService.selectDictDataByType("t_trouble_type");
                    for (SysDictData sysDictData : sysDictDataList) {
                        if (temp.getTroubleType().equals(sysDictData.getDictValue())) {
                            hiddenTroubleVo.setTroubleType(sysDictData.getDictLabel());
                        }
                    }
                }
                if (StringUtils.isNotEmpty(temp.getTroubleLevel())) {
                    List<SysDictData> sysDictDataList = iSysDictTypeService.selectDictDataByType("t_trouble_level");
                    for (SysDictData sysDictData : sysDictDataList) {
                        if (temp.getTroubleLevel().equals(sysDictData.getDictValue())) {
                            hiddenTroubleVo.setTroubleLevel(sysDictData.getDictLabel());
                        }
                    }
                }
                if (StringUtils.isNotEmpty(temp.getDealStatus())) {
                    List<SysDictData> sysDictDataList = iSysDictTypeService.selectDictDataByType("t_deal_status");
                    for (SysDictData sysDictData : sysDictDataList) {
                        if(temp.getDealStatus() == null){
                            hiddenTroubleVo.setDealStatus("暂未处理");
                        } else if (temp.getDealStatus().equals(sysDictData.getDictValue())) {
                            hiddenTroubleVo.setDealStatus(sysDictData.getDictLabel());
                        }
                    }
                }

                list.add(hiddenTroubleVo);
            }
        }
        return list;
    }

    /**
     * 新增隐患信息
     * 
     * @param tHiddenTrouble 隐患信息
     * @return 结果
     */
    @Override
    public int insertTHiddenTrouble(THiddenTrouble tHiddenTrouble)
    {
        tHiddenTrouble.setCreateTime(DateUtils.getNowDate());
        return tHiddenTroubleMapper.insertTHiddenTrouble(tHiddenTrouble);
    }

    /**
     * 修改隐患信息
     * 
     * @param tHiddenTrouble 隐患信息
     * @return 结果
     */
    @Override
    public int updateTHiddenTrouble(THiddenTrouble tHiddenTrouble)
    {
        tHiddenTrouble.setUpdateTime(DateUtils.getNowDate());
        return tHiddenTroubleMapper.updateTHiddenTrouble(tHiddenTrouble);
    }

    /**
     * 批量删除隐患信息
     * 
     * @param troubleIds 需要删除的隐患信息ID
     * @return 结果
     */
    @Override
    public int deleteTHiddenTroubleByIds(int[] troubleIds)
    {
        return tHiddenTroubleMapper.deleteTHiddenTroubleByIds(troubleIds);
    }

    /**
     * 删除隐患信息信息
     * 
     * @param troubleId 隐患信息ID
     * @return 结果
     */
    @Override
    public int deleteTHiddenTroubleById(int troubleId)
    {
        return tHiddenTroubleMapper.deleteTHiddenTroubleById(troubleId);
    }
}
