package com.zehong.web.controller.workOrder;

import java.util.List;

import com.zehong.common.utils.StringUtils;
import com.zehong.system.domain.vo.WorkOrderVo;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.enums.BusinessType;
import com.zehong.system.domain.TWorkOrder;
import com.zehong.system.service.ITWorkOrderService;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.common.core.page.TableDataInfo;

/**
 * 工单基础信息Controller
 * 
 * @author zehong
 * @date 2021-07-19
 */
@RestController
@RequestMapping("/workOrder/basicsInfo")
public class TWorkOrderController extends BaseController
{
    @Autowired
    private ITWorkOrderService tWorkOrderService;

    /**
     * 查询工单基础信息列表
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:list')")
    @GetMapping("/list")
    public TableDataInfo list(TWorkOrder tWorkOrder)
    {
        startPage();
        List<WorkOrderVo> list = tWorkOrderService.selectTWorkOrderList(tWorkOrder);
        return getDataTable(list);
    }

    /**
     * 导出工单基础信息列表
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:export')")
    @Log(title = "工单基础信息", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TWorkOrder tWorkOrder)
    {
        List<WorkOrderVo> list = tWorkOrderService.selectTWorkOrderList(tWorkOrder);
        ExcelUtil<WorkOrderVo> util = new ExcelUtil<WorkOrderVo>(WorkOrderVo.class);
        return util.exportExcel(list, "工单基础信息数据");
    }

    /**
     * 获取工单基础信息详细信息
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:query')")
    @GetMapping(value = "/{orderId}")
    public AjaxResult getInfo(@PathVariable("orderId") String orderId)
    {
        return AjaxResult.success(tWorkOrderService.selectTWorkOrderById(orderId));
    }

    /**
     * 新增工单基础信息
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:add')")
    @Log(title = "工单基础信息", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TWorkOrder tWorkOrder)
    {
        return toAjax(tWorkOrderService.insertTWorkOrder(tWorkOrder));
    }

    /**
     * 修改工单基础信息
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:edit')")
    @Log(title = "工单基础信息", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TWorkOrder tWorkOrder) throws Exception {

        String orderStatus = tWorkOrder.getOrderStatus();
        // 工单状态只有是0未下发或1已下发,才允许修改
        if(!"0".equals(orderStatus) && !"1".equals(orderStatus)){
            throw new Exception(StringUtils.format("工单({})当前状态不允许修改。", tWorkOrder.getOrderId()));
        }

        return toAjax(tWorkOrderService.updateTWorkOrder(tWorkOrder));
    }

    /**
     * 修改工单状态
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:editStatus')")
    @Log(title = "工单基础信息", businessType = BusinessType.UPDATE)
    @PutMapping("/editStatus")
    public AjaxResult editStatus(@RequestBody TWorkOrder tWorkOrder) throws Exception {

        String orderStatus = tWorkOrder.getOrderStatus();
        if("5".equals(orderStatus)) {
            throw new Exception(StringUtils.format("工单({})已归档，不允许更改状态。", tWorkOrder.getOrderId()));
        }
        return AjaxResult.success();
    }

    /**
     * 删除工单基础信息
     */
    @PreAuthorize("@ss.hasPermi('workOrder:basicsInfo:remove')")
    @Log(title = "工单基础信息", businessType = BusinessType.DELETE)
	@DeleteMapping("/{orderIds}")
    public AjaxResult remove(@PathVariable String[] orderIds)
    {
        return toAjax(tWorkOrderService.deleteTWorkOrderByIds(orderIds));
    }
}
