package com.zehong.web.controller.device;

import com.zehong.common.annotation.Log;
import com.zehong.common.core.controller.BaseController;
import com.zehong.common.core.domain.AjaxResult;
import com.zehong.common.core.page.TableDataInfo;
import com.zehong.common.enums.BusinessType;
import com.zehong.common.utils.poi.ExcelUtil;
import com.zehong.system.domain.TDeviceInfo;
import com.zehong.system.domain.vo.DeviceInfoVo;
import com.zehong.system.service.ITDeviceInfoService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 设备信息Controller
 * 
 * @author zehong
 * @date 2021-07-09
 */
@RestController
@RequestMapping("/device/deviceInfo")
public class TDeviceInfoController extends BaseController
{
    @Autowired
    private ITDeviceInfoService tDeviceInfoService;

    /**
     * 查询设备信息列表
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:list')")
    @GetMapping("/list")
    public TableDataInfo list(TDeviceInfo tDeviceInfo)
    {
        startPage();
        List<DeviceInfoVo> list = tDeviceInfoService.selectTDeviceInfoList(tDeviceInfo);
        return getDataTable(list);
    }

    /**
     * 获取所有设备信息
     * @param tDeviceInfo
     * @return
     */
    @GetMapping("/deviceListInfo")
    public AjaxResult deviceListInfo(TDeviceInfo tDeviceInfo){
        return AjaxResult.success(tDeviceInfoService.selectTDeviceInfoList(tDeviceInfo));
    }

    /**
     * 获取设备下拉树列表
     */
    @PutMapping("/deviceTree")
    public AjaxResult deviceTree(@RequestBody Map<Object, List> param)
    {
        return AjaxResult.success(tDeviceInfoService.buildDeviceTreeSelect(param));
    }

    /**
     * 获取设备树
     */
    @GetMapping("/deviceNodeTree")
    public AjaxResult deviceNodeTree() throws Exception
    {
        List<Map<Object, Object>> list = null;
        try {
            list = tDeviceInfoService.buildDeviceTree();
        } catch (Exception e) {
            e.printStackTrace();
            throw new Exception("获取设备列表失败");
        }

        return AjaxResult.success(list);
    }

    /**
     * 导出设备信息列表
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:export')")
    @Log(title = "设备信息", businessType = BusinessType.EXPORT)
    @GetMapping("/export")
    public AjaxResult export(TDeviceInfo tDeviceInfo)
    {
        List<DeviceInfoVo> list = tDeviceInfoService.selectTDeviceInfoList(tDeviceInfo);
        ExcelUtil<DeviceInfoVo> util = new ExcelUtil<DeviceInfoVo>(DeviceInfoVo.class);
        return util.exportExcel(list, "设备信息数据");
    }

    /**
     * 获取设备信息详细信息
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:query')")
    @GetMapping(value = "/{deviceId}")
    public AjaxResult getInfo(@PathVariable("deviceId") int deviceId)
    {
        return AjaxResult.success(tDeviceInfoService.selectTDeviceInfoById(deviceId));
    }

    /**
     * 新增设备信息
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:add')")
    @Log(title = "设备信息", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TDeviceInfo tDeviceInfo)
    {
        int result = tDeviceInfoService.insertTDeviceInfo(tDeviceInfo);
        if(result<=0){
            new Exception("设备信息插入失败");
        }
        return AjaxResult.success(tDeviceInfo.getDeviceId());
    }

    /**
     * 修改设备信息
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:edit')")
    @Log(title = "设备信息", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TDeviceInfo tDeviceInfo)
    {
        return toAjax(tDeviceInfoService.updateTDeviceInfo(tDeviceInfo));
    }

    /**
     * 删除设备信息
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:remove')")
    @Log(title = "设备信息", businessType = BusinessType.DELETE)
	@DeleteMapping("/{deviceIds}")
    public AjaxResult remove(@PathVariable int[] deviceIds)
    {
        return toAjax(tDeviceInfoService.deleteTDeviceInfoByIds(deviceIds));
    }

    /**
     * 统计各设备类型的设备总数
     */
    @PreAuthorize("@ss.hasPermi('device:deviceInfo:countDeviceByType')")
    @GetMapping("/countDeviceByType")
    public AjaxResult countDeviceByType()
    {
        List<Map<Object, Object>> list = tDeviceInfoService.countDeviceByType();
        return AjaxResult.success(list);
    }
}
